package name.matthewgreet.strutscommons.interceptor;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.form.Form;
import name.matthewgreet.strutscommons.util.AnnotationValidatior;
import name.matthewgreet.strutscommons.util.DefaultAnnotationValidatior;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.ModelDriven;
import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.interceptor.MethodFilterInterceptor;
import com.opensymphony.xwork2.interceptor.ValidationAware;

/**
 * <P>Struts 2 Interceptor for setting form fields from request parameters, adjusting, converting, validating, and 
 * writing error messages where data type conversion fails, according to form field annotations.  If the Action 
 * implements {@link ModelDriven}, the model is recognised as the form, not the action.  Struts Actions are expected to 
 * implement {@link ValidationAware} and, if message keys are used, implement {@link TextProvider}.</P>
 * 
 * <P>Annotations are applied to a form's member variables.  The standard set are found in 
 * {@link name.matthewgreet.strutscommons.annotation} and includes customisable ones.  Annotations configure various 
 * kinds of policies and typically don't apply for missing form field values (@Required being a notable exception).</P>
 * <TABLE>
 *   <CAPTION>Types of Annotations</CAPTION>
 *   <TR>
 *     <TH>Policy</TH>
 *     <TH>Example</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Adjuster</TD>
 *     <TD>@ToUpperCase</TD>
 *     <TD>Modifies form field value before validation or conversion.</TD>
 *   </TR>
 *   <TR>
 *     <TD>Non-conversion validator</TD>
 *     <TD>@MaxLength</TD>
 *     <TD>Validates form field value before any conversion.  Usually only applicable to string fields, with @Required 
 *         as a notable exception.</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Converter</TD>
 *     <TD>@DateConversion</TD>
 *     <TD>Converts request parameter from string to other data type.</TD>
 *   </TR>
 *   <TR>
 *     <TD>Post-conversion validator</TD>
 *     <TD>@IntegerRange</TD>
 *     <TD>Validates converted form field value.</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Manual parameter conversion</TD>
 *     <TD>@ManualParameterConversion</TD>
 *     <TD>A special annotation indicating form field is processed in manual parameter conversion mode (see below).</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>Each form field is processed in one of the modes described below.</P>
 * <TABLE>
 *   <CAPTION>Validation Modes</CAPTION>
 *   <TR>
 *     <TH>Mode</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR>
 *     <TD>Auto conversion</TD>
 *     <TD>Directly sets form field if it passes non-conversion validation and conversion.</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Default conversion</TD>
 *     <TD>Same as auto conversion but a default converter is used for the field's data type.</TD>
 *   </TR>
 *   <TR>
 *     <TD>Manual parameter conversion</TD>
 *     <TD>Adjustment, conversion, and validation is done by form code.  Form must implement {@link Form}.</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Pair conversion</TD>
 *     <TD>Sets a string form field and applies adjusters and validators and converts to a paired, non-string form 
 *         field.</TD>
 *   </TR>
 *   <TR>
 *     <TD>No conversion</TD>
 *     <TD>Sets a string form field and applies adjusters and validators.</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *     <TD>Set only</TD>
 *     <TD>Sets a string form field and no policies are applied.</TD>
 *   </TR>
 * </TABLE>
 * <P>Except where a @ManualParameterConversion annotation is used, how each mode applies is summarised below.</P>
 * <TABLE>
 *   <CAPTION>Mode Usage</CAPTION>
 *   <TR>
 *     <TH>Field type</TH>
 *     <TH>Converter annotation</TH>
 *     <TH>Mode</TH>
 *     <TH>Notes</TH>
 *   </TR>
 *   <TR>
 *     <TD>String</TD>
 *     <TD>No</TD>
 *     <TD>No conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR>
 *     <TD>String</TD>
 *     <TD>Yes</TD>
 *     <TD>Pair conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>Non-string</TD>
 *           <TD>No</TD>
 *           <TD>Default conversion</TD>
 *           <TD></TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>Non-string</TD>
 *           <TD>Yes</TD>
 *           <TD>Auto conversion</TD>
 *           <TD></TD>
 *   </TR>
 *   <TR>
 *           <TD>String array</TD>
 *           <TD>&nbsp;</TD>
 *           <TD>Set only</TD>
 *           <TD>Converters don't apply to arrays</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>String collection</TD>
 *           <TD>No</TD>
 *           <TD>Set only</TD>
 *           <TD></TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>String collection</TD>
 *           <TD>Yes</TD>
 *           <TD>Auto conversion</TD>
 *           <TD></TD>
 *   </TR>
 *   <TR>
 *           <TD>Non-string array</TD>
 *           <TD>&nbsp;</TD>
 *           <TD>Manual parameter conversion</TD>
 *           <TD>Converters don't apply to arrays</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>Non-string collection</TD>
 *           <TD>No</TD>
 *           <TD>Default conversion</TD>
 *           <TD>Few default converters for collection types exist</TD>
 *   </TR>
 *   <TR CLASS="even_row">
 *           <TD>Non-string collection</TD>
 *           <TD>Yes</TD>
 *           <TD>Auto conversion</TD>
 *           <TD></TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>This replaces the standard AnnotationValidationInterceptor, ConversionErrorInterceptor and ParametersInterceptor 
 * and differs in various aspects.</P>
 * <UL>
 *   <LI>Unlike ParametersInterceptor, directly sets form fields, rather than setting properties on the Value Stack.</LI>
 *   <LI>Unlike ParametersInterceptor, does not convert multiple request parameters with the same name.  See above.</LI>
 *   <LI>Unlike ConversionErrorInterceptor, error messages aren't restricted to field errors.</LI>
 * </UL>
 * 
 * <P><u>Interceptor parameters:</u></P>
 *
 * <DL>
 *  <DT>disabled</DT>
 *  <DD>If true, this interceptor is skipped.</DD>
 *  <DT>paramNameMaxLength</DT>
 *  <DD>Ignores all request parameters with names longer than this.</DD>
 *  <DT>processUnreferencedFields</DT>
 *  <DD>If true, does not skip fields that have no policy annotations and aren't set by a parameter.  This will only be 
 *      useful where a default converter must still process missing values.</DD>
 * </DL>
 *
 * <P><U>Extending the interceptor:</U></P>
 *
 * <P>Various points of the core algorithm can be overridden to customize behaviour.  See javadoc of protected methods.</P>
 *
 * <P> <u>Example code:</u></P>
 * <PRE>
 * &lt;action name="someAction" class="com.examples.SomeAction"&gt;
 *     &lt;interceptor-ref name="annotationValidation"/&gt;
 *     &lt;interceptor-ref name="validation"/&gt;
 *     &lt;result name="success"&gt;good_result.ftl&lt;/result&gt;
 * &lt;/action&gt;
 * <!-- END SNIPPET: example -->
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class AnnotationValidationInterceptor2 extends MethodFilterInterceptor {
    private static final long serialVersionUID = 2689404505465349761L;
    
    protected static final int PARAM_NAME_MAX_LENGTH = 100;
    

    private boolean disabled;

    
    @SuppressWarnings("unused")
	private Logger LOG = LogManager.getLogger(AnnotationValidationInterceptor2.class);
    
    private int paramNameMaxLength = PARAM_NAME_MAX_LENGTH;
    private boolean processUnreferencedFields = false;
    

    @Override
    protected String doIntercept(ActionInvocation invocation) throws Exception {
    	AnnotationValidatior annotationValidationLibrary;
    	
        if (!disabled) {
        	annotationValidationLibrary = makeAnnotationValidationLibrary();
        	annotationValidationLibrary.validate();
        }
        
        return invocation.invoke();
    }

	protected AnnotationValidatior makeAnnotationValidationLibrary() {
		DefaultAnnotationValidatior result;
		
		result = new DefaultAnnotationValidatior();
		result.setParamNameMaxLength(getParamNameMaxLength());
		result.setProcessUnreferencedFields(processUnreferencedFields);
		return result;
	}

    
	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	
	public int getParamNameMaxLength() {
		return paramNameMaxLength;
	}
	public void setParamNameMaxLength(int paramNameMaxLength) {
		this.paramNameMaxLength = paramNameMaxLength;
	}

	public boolean getProcessUnreferencedFields() {
		return processUnreferencedFields;
	}
	public void setProcessUnreferencedFields(boolean processUnreferencedFields) {
		this.processUnreferencedFields = processUnreferencedFields;
	}


}
