package name.matthewgreet.strutscommons.interceptor;




import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;
import com.opensymphony.xwork2.interceptor.ValidationAware;


/**
 * <P>Detects an exception not handled by the form driven action, logs it, changes the result to 'input' or 'error' as 
 * needed and, if it implements ValidationAware, writes an action error for the user.  Checked exceptions are regarded 
 * as an application rejection and use the 'input' result.  Unchecked exceptions are regarded as system failures and use 
 * the 'error' result.</P>
 * 
 * <P><U>Interceptor parameters:</U></P>
 *
 * <DL>
 *  <DT>disabledChangeResult</DT>
 *  <DD>If true, does not change result and rethrows exception without logging it.  Defaults to false.</DD>
 *  <DT>disabledErrorMessage</DT>
 *  <DD>If true, does not write error message to user.  Defaults to false.</DD>
 *  <DT>applicationErrorMessage</DT>
 *  <DD>Error message to write for application rejection with the exception object replacing '${e}'.  Defaults to 
 *      "The action was rejected: ${e}".</DD>
 *  <DT>systemFailureErrorMessage</DT>
 *  <DD>Error message to write for system failures'.  Defaults to "A system failure occurred, which was logged".</DD>
 * </DL>
 */
@SuppressWarnings("deprecation")
public class UnhandledExceptionWriterInterceptor extends AbstractInterceptor {
    private static final long serialVersionUID = -1257373050470853612L;
    
    private Logger logger;
    
    private boolean disabledChangeResult = false;
    private boolean disabledErrorMessage = false;
    private String applicationErrorMessage = "The action was rejected: ${e}";
    private String systemFailureErrorMessage = "A system failure occurred, which was logged";
    
    
    public UnhandledExceptionWriterInterceptor() {
        super();
    }
    
    /**
     * Ensures the logger for this instance has been initialised. 
     */
    protected void checkLogger(ActionInvocation invocation) {
        if (logger == null) {
            logger = makeLogger(invocation);
        }
    }
    
    /**
     * Returns logger to be used for this instance; 
     */
    protected Logger getLogger() {
        return logger;
    }

    /**
     * Returns whether exception indicates a system failure, rather than rejection due to business rules.  
     */
    protected boolean isSystemFailureException(Exception e) {
        return e instanceof RuntimeException;
    }
    
    /**
     * Called to log exception before diverting to new result.  
     */
    protected void logException(ActionInvocation invocation, boolean isSystemFailure, Exception e) {
        String message;
        
        if (isSystemFailure) {
            message = "Uncaught system failure";
        } else {
            message = "Uncaught application rejection";
        }
        getLogger().error(message, e);
    }

    /**
     * Returns a new logger to be used by this interceptor instance. 
     */
    protected Logger makeLogger(ActionInvocation invocation) {
        return LogManager.getLogger(invocation.getAction().getClass().getName());
    }
    
    @Override
    public String intercept(ActionInvocation invocation) throws Exception {
        ValidationAware validationAware;
        String errorMessage;
        boolean isSystemFailure;
        
        try {
            return invocation.invoke();
        }
        catch (Exception e) {
            isSystemFailure = isSystemFailureException(e);
            if (!disabledErrorMessage && invocation.getAction() instanceof ValidationAware) {
                validationAware = (ValidationAware)invocation.getAction();
                if (isSystemFailure) {
                    errorMessage = systemFailureErrorMessage;
                } else {
                    errorMessage = applicationErrorMessage;
                }
                errorMessage = errorMessage.replace("${e}", (e.getMessage() != null)?e.getMessage():"");
                validationAware.addActionError(errorMessage);
            }
            if (!disabledChangeResult) {
                checkLogger(invocation);
                logException(invocation, isSystemFailure, e);
                if (isSystemFailure) {
                    return "error";
                } else {
                    return "input";
                }
            } else {
                throw e;
            }
        }
    }

    /**
     * Returns whether changing result and logging when an exception is caught is disabled.  Defaults to false;
     */
    public boolean getDisabledChangeResult() {
        return disabledChangeResult;
    }
    public void setDisabledChangeResult(boolean disabledChangeResult) {
        this.disabledChangeResult = disabledChangeResult;
    }

    /**
     * Returns whether writing error message is disabled.  Defaults to false.
     */
    public boolean getDisabledErrorMessage() {
        return disabledErrorMessage;
    }

    public void setDisabledErrorMessage(boolean disabledErrorMessage) {
        this.disabledErrorMessage = disabledErrorMessage;
    }

    /**
     * Returns error message for application exceptions.  May contain '${e}' as substitution field for exception's 
     * message.  Defaults to "The action was rejected: ${e}".
     */
    public String getApplicationErrorMessage() {
        return applicationErrorMessage;
    }
    public void setApplicationErrorMessage(String applicationErrorMessage) {
        this.applicationErrorMessage = applicationErrorMessage;
    }

    /**
     * Returns error message for system failure exceptions.  May contain '${e}' as substitution field for exception's 
     * message.  Defaults to "A system failure occurred, which was logged".
     */
    public String getSystemFailureErrorMessage() {
        return systemFailureErrorMessage;
    }
    public void setSystemFailureErrorMessage(String systemFailureErrorMessage) {
        this.systemFailureErrorMessage = systemFailureErrorMessage;
    }

}
