package name.matthewgreet.strutscommons.view;

import java.util.*;


/**
 * Model bean, passed to JSP, that stores list and index of currently, selected item.
 */
public class FormattedListDisplay<T> {
    private List<T> list;
    private boolean pageMode;
    private int page;           // Starts at 1
    private int pageSize;
    private int totalItems;
    private int totalPages;
    private int selectedIndex;  // Starts at 0, like the rest of Java

    /**
     * Default constructor. 
     */
    public FormattedListDisplay() {
        super();
    }

    /**
     * Returns index of last item in current page relative to master list that backs it (which starts at index 0).
     * This is -1 if there are no entries, or list size - 1 if not in page mode.
     */
    public int getEndMasterIndex() {
        if (pageMode && page == totalPages) {
            return totalItems - 1;
        } else if (pageMode && list.size() > 0) {
            return page * pageSize - 1;
        } else if (list.size() == 0) {
            return -1;
        } else {
            return list.size() - 1;
        }
    }
    
    /**
     * Returns whether there is a currently selected item. 
     */
    public boolean getHasSelectedItem() {
        return selectedIndex >= 0;
    }
    
    /**
     * Returns list for display, which may be a page if in page mode.   
     */
    public List<T> getList() {
        return list;
    }
    public void setList(List<T> value) {
        list = value;
    }
    
    
    /**
     * Returns size of list.  This exists because JSP EL can only read data if presented as a Java bean property. 
     */
    public int getListSize() {
        return list.size();
    }
    
    /**
     * Whether list should be displayed in pages.   
     */
    public boolean getPageMode() {
        return pageMode;
    }
    public void setPageMode(boolean value) {
        pageMode = value;
    }
    
    /**
     * Returns current page number, starting at 1.  Only useful if in page mode.   
     */
    public int getPage() {
        return page;
    }
    public void setPage(int value) {
        page = value;
    }
    
    /**
     * Returns index of last item in current page of list (which starts at index 0).  This is -1 if there are no 
     * entries, or list size - 1 if not in page mode.
     */
    public int getPageEndIndex() {
        if (pageMode && page == totalPages) {
            return totalItems - 1;
        } else if (pageMode && list.size() > 0) {
            return page * pageSize - 1;
        } else if (list.size() == 0) {
            return -1;
        } else {
            return list.size() - 1;
        }
    }
    
    /**
     * Returns size of pages.  Only useful if in page mode.   
     */
    public int getPageSize() {
        return pageSize;
    }
    public void setPageSize(int value) {
        pageSize = value;
    }
    

    /**
     * Returns index of first item in current page of list (which starts at index 0).  This is -1 if there are no
     * entries, or 0 if not in page mode.
     */
    public int getPageStartIndex() {
        if (pageMode && list.size() > 0) {
            return (page - 1) * pageSize;
        } else if (list.size() == 0) {
            return -1;
        } else {
            return 0;
        }
    }

    /**
     * Returns total number of items.   
     */
    public int getTotalItems() {
        return totalItems;
    }
    public void setTotalItems(int totalItems) {
        this.totalItems = totalItems;
    }

    /**
     * Returns total number of pages.  Only useful if in page mode.   
     */
    public int getTotalPages() {
        return totalPages;
    }
    public void setTotalPages(int value) {
        totalPages = value;
    }
    
    /**
     * Returns index of currently selected item relative to current page, stating at 0. 
     */
    public int getSelectedIndex() {
        return selectedIndex;
    }
    public void setSelectedIndex(int value) {
        selectedIndex = value;
    }
    
    /**
     * Returns index of currently selected item in list cache that backs the current page, starting at 0.  This is the 
     * same as the selection index if not in page mode. 
     */
    public int getSelectedMasterIndex() {
        if (pageMode) {
            return (page - 1) * pageSize + selectedIndex;
        } else {
            return selectedIndex;
        }
    }
    
    /**
     * Returns currently selected item in list. 
     */
    public T getSelectedItem() {
        if (selectedIndex >= 0) {
            return list.get(selectedIndex);
        } else {
            return null;
        }
    }
    
    /**
     * Returns index of first item in current page relative to master list that backs it (which starts at index 0).
     * This is -1 if there are no entries, or 0 if not in page mode.
     */
    public int getStartMasterIndex() {
        if (pageMode && list.size() > 0) {
            return (page - 1) * pageSize;
        } else if (list.size() == 0) {
            return -1;
        } else {
            return 0;
        }
    }

}
