package name.matthewgreet.strutscommons.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.conversion.impl.ConversionData;


/**
 * Template that aids creation of OPTION tags in JSPs for single selection SELECT elements by formatting an unformatted 
 * list for display.  Concrete subclasses override {@link #getKey}, {@link #getValue} and {@link #getText} to extract 
 * record ids, OPTION tag values and display text from an unformatted list.  See also {@link SelectBoxItemDisplay2}.
 * 
 * <P>Example JSP code:<BR/>
 * <CODE>
 * &lt;SELECT SIZE="1" NAME="category" /&gt;<BR/>
 * &nbsp;&nbsp;&lt;s:iterator value="categoryDisplay.list" var="categoryItemDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTION VALUE="&lt;s:property value="#categoryItemDisplay.value"/&gt;" &lt;s:property value="#categoryItemDisplay.selectedAttribute"/&gt;&gt;&lt;s:property value="#categoryItemDisplay.text"/&gt;&lt;/OPTION&gt;<BR/>
 * &nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &lt;/SELECT&gt;<BR/>
 * </CODE></P>
 */
@SuppressWarnings("deprecation")
public abstract class SingleSelectBoxDisplay2<K,T> {
    public class TextComparator implements Comparator<SelectBoxItemDisplay2<K,T>> {
        @Override
        public int compare(SelectBoxItemDisplay2<K,T> o1, SelectBoxItemDisplay2<K,T> o2) {
            return o1.getText().compareToIgnoreCase(o2.getText());
        }
    }

    private static Logger LOG = LogManager.getLogger(SingleSelectBoxDisplay2.class);
    
    private List<SelectBoxItemDisplay2<K,T>> formattedModel;
    private int selectedIndex;
    
    public SingleSelectBoxDisplay2() {
        formattedModel = getInitialList();
        if (formattedModel == null) {
            formattedModel = new ArrayList<SelectBoxItemDisplay2<K,T>>();
        }
        selectedIndex = -1;
    }

    /**
     * May be overridden by subclasses to add additional, formatted items to the display list.  This is typically used 
     * to add a blank item to represent no value or text that demands a real value is selected.  This is called after 
     * sorting.
     * 
     * @param formattedModel List of {@link SelectBoxItemDisplay}
     */
    protected void addItems(List<SelectBoxItemDisplay2<K,T>> formattedModel) {
        // Empty
    }
    
    /**
     * If {@link #hasBlankValue} returns true or {@link #addBlankValue} is called, returns text to display for blank 
     * value.  Defaults to blank.
     */
    protected String getBlankText() {
        return "";
    }
    
    /**
     * May be overridden by subclasses to return initial selection, especially if the list is hardcoded and setModel 
     * will not be used.  Can return null for no initial list.
     */
    protected List<SelectBoxItemDisplay2<K,T>> getInitialList() {
        return null;
    }
    
    /**
     * Overridden by subclasses to return unformatted identifier of item record.  This is the value used to match the 
     * parameter of {@link #setSelectedValue}.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract K getKey(T item);
    
    /**
     * May be overridden by subclasses to return logger of subclass.
     */
    protected Logger getLogger() {
        return LOG;
    }
    
    /**
     * May be overridden by subclasses to return a comparator for defining display order of formatted items.  Comparator
     * compares instances of {@link SelectBoxItemDisplay} that are allowed by subclass.  Defaults to comparator that 
     * sorts by ascending display text. 
     */
    protected Comparator<SelectBoxItemDisplay2<K,T>> getSortComparator() {
        return new TextComparator();
    }
    
    /**
     * Overridden by subclasses to return formatted text to be displayed to user as part of OPTION element.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getText(T item);
    
    /**
     * Overridden by subclasses to return string to be used in VALUE attribute of OPTION element.  This is the formatted 
     * version of the item record identifier and is the value used to match the parameter of 
     * {@link #setSelectedFormattedValue}.
     *  
     * @param key Identifier of unformatted record.
     */
    protected abstract String getValue(K key);
    
    /**
     * May be overridden by subclasses to return true, inserting a blank value at the beginning of the SELECT list.
     * Defaults to false. 
     */
    protected boolean hasBlankValue() {
        return false;
    }
    
    /**
     * May be overridden by subclasses to filter items in lookup list from display.  Defaults to allow all items.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected boolean isAllowed(T item) {
        return true;
    }
    
    /**
     * If called after {@link #setModel}, inserts blank value to beginning of list.  This is useful for search fields 
     * where a blank value means no search.
     */
    public void addBlankValue() {
        SelectBoxItemDisplay2<K,T> formattedItem;
        
        formattedItem = new SelectBoxItemDisplay2<K,T>("", getBlankText(), null, null);
        formattedModel.add(0, formattedItem);
    }
    
    /**
     * Returns formatted version of account list for human display.
     */
    public List<SelectBoxItemDisplay2<K,T>> getList() {
        return formattedModel;
    }
    
    /**
     * Returns selected, formatted item or null if none selected. 
     */
    public SelectBoxItemDisplay2<K,T> getSelectedItem() {
        if (selectedIndex > -1) {
            return formattedModel.get(selectedIndex);
        } else {
            return null;
        }
    }
    
    /**
     * Directly sets formatted list.  The value must already be sorted.
     */
    public void setList(List<SelectBoxItemDisplay2<K,T>> value) {
        formattedModel = value;
    }
    
    /**
     * Sets unformatted values and creates formatted, sorted list from it.
     */
    public void setModel(Collection<T> model) {
        SelectBoxItemDisplay2<K,T> formattedItem;
        K key;
        String value, text;
        
        try {
            formattedModel.clear();
            if (model != null) {
                for (T item: model) {
                    if (isAllowed(item)) {
                    	key = getKey(item);
                        value = getValue(key);
                        text = getText(item);
                        formattedItem = new SelectBoxItemDisplay2<K,T>(value, text, key, item);
                        formattedModel.add(formattedItem);
                    }
                }
            }
            
            Collections.sort(formattedModel, getSortComparator());
            
            addItems(formattedModel);
            if (hasBlankValue()) {
                formattedModel.add(0, new SelectBoxItemDisplay2<K,T>("", getBlankText(), null, null));
            }
        }
        catch (Exception e) {
            LOG.error("Setup of JSP view helper failed", e);
        }
    }
    
    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or formatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param value Formatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedFormattedValueWithConversionError(String value, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedFormattedValue(value);
		}
    }

    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or unformatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param key Unformatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedValueWithConversionError(K key, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedValue(key);
		}
    }

    /**
     * Selects item matching formatted identifier.  This matches the OPTION tag VALUE attribute.
     * 
     * @param value Formatted identifier of selected item or null or empty string to select none.
     */
    public void setSelectedFormattedValue(String value) {
        SelectBoxItemDisplay2<K,T> item;
        ListIterator<SelectBoxItemDisplay2<K,T>> iter;
        boolean found;
        
        // Deselect existing, selected item.
        if (selectedIndex > -1) {
            item = formattedModel.get(selectedIndex);
            item.setSelected(false);
        }
        selectedIndex = -1;
        
        if (value != null && value.length() > 0) {
            // Find item and set it as selected
            iter = formattedModel.listIterator();
            found = false;
            while (iter.hasNext() && !found) {
                item = iter.next();
                if (item.getValue().equals(value)) {
                    selectedIndex = iter.previousIndex();
                    item.setSelected(true);
                    found = true;
                }
            }
        }
    }
    /**
     * Selects item matching unformatted identifier.  Formatted version of key matches the OPTION tag VALUE attribute.
     * 
     * @param key id of selected item or null to select none.
     */
    public void setSelectedValue(K key) {
        SelectBoxItemDisplay2<K,T> item;
        ListIterator<SelectBoxItemDisplay2<K,T>> iter;
        boolean found;
        
        // Deselect existing, selected item.
        if (selectedIndex > -1) {
            item = formattedModel.get(selectedIndex);
            item.setSelected(false);
        }
        selectedIndex = -1;
        
        if (key != null) {
            // Find item and set it as selected
            iter = formattedModel.listIterator();
            found = false;
            while (iter.hasNext() && !found) {
                item = iter.next();
                if (key.equals(item.getKey())) {
                    selectedIndex = iter.previousIndex();
                    item.setSelected(true);
                    found = true;
                }
            }
        }
    }
}
