package name.matthewgreet.strutscommons.junit.policy;

import java.util.Arrays;
import java.util.Collection;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

import name.matthewgreet.strutscommons.annotation.IntegerCSVConversion;
import name.matthewgreet.strutscommons.policy.IntegerCSVConverter;
import name.matthewgreet.strutscommons.policy.ConversionResult;


public class IntegerCSVConverterUnitTests {
	@IntegerCSVConversion
	private boolean annotationPlaceholderDefault;
	
	@IntegerCSVConversion(separator = ":")
	private boolean annotationPlaceholderSeparator;
	
	
	@BeforeEach
	public void init() {
		// Empty
	}
	
	
	@DisplayName("Normal conversion")
	@Test
	public void testConvert1() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("334579,34,-987", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(334579,34,-987), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Failed conversion")
	@Test
	public void testConvert2() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerConverter = new IntegerCSVConverter();
		integerConverter.setAnnotation(annotation);
		result = integerConverter.convert("$334579,34,-987", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(null, result.getMessage(), "Standard policies leave message to annotation");
		Assertions.assertEquals(null, result.getMessageKey(), "Standard policies leave message key to annotation");
		Assertions.assertEquals(null, result.getMessageType(), "Standard policies leave message type to annotation");
		Assertions.assertEquals(false, result.getSuccess(), "Conversion should have failed");
	}
    
	@DisplayName("Alternate separator")
	@Test
	public void testConvert3() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderSeparator").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("334579:34:-987", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(334579,34,-987), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Contains empty value")
	@Test
	public void testConvert4() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("334579,,-987", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(334579,null,-987), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Starts with empty value")
	@Test
	public void testConvert5() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert(",4467,321", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(null,4467,321), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Ends with empty value")
	@Test
	public void testConvert6() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("334579,4467,", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(334579,4467,null), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Single value")
	@Test
	public void testConvert7() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("42", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(42), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Empty value")
	@Test
	public void testConvert8() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		ConversionResult<Integer> result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.convert("", Collection.class, Integer.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertIterableEquals(Arrays.asList(), result.getParsedCollectionValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Normal format")
	@Test
	public void testFormat1() throws Exception {
		IntegerCSVConversion annotation;
		IntegerCSVConverter integerCSVConverter;
		String result;
		
		annotation = (IntegerCSVConversion)IntegerCSVConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(IntegerCSVConversion.class);
		integerCSVConverter = new IntegerCSVConverter();
		integerCSVConverter.setAnnotation(annotation);
		result = integerCSVConverter.format(Arrays.asList(560009, -892835, 4467));
		Assertions.assertEquals("560009,-892835,4467", result, "Final value is incorrect");
	}
    
}
