package name.matthewgreet.strutscommons.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.policy.AbstractCustomCollectionConverterSupport;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.util.StrutsMiscellaneousLibrary;

/**
 * <P>Defines a form field (whether on a Struts Action or model of ModelDriven Struts Action) should be converted 
 * using a client supplied collection converter to a recipient field of some type of collection and error message to 
 * display if this fails.  The custom collection converter must implement {@link CollectionConverter}, preferably 
 * extending {@link AbstractCustomCollectionConverterSupport}, and have a default public constructor.  Also, defines a 
 * view helper field receives a sting formated by the custom collection converter.</P>
 * 
 * <P>There are two modes of behaviour for conversion, depending on the type of form field.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Annotation Usage</CAPTION>
 *   <TR>
 *     <TH>Form field type</TH>
 *     <TH>Mode</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String</TD>
 *     <TD>Pair conversion</TD>
 *     <TD>Converts string field value to collection type field named by parsedFieldName</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Collection type</TD>
 *     <TD>Auto conversion</TD>
 *     <TD>Converts directly from request parameter</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>By default, messages are added to the actionErrors list but this can be overridden by <CODE>messageType</CODE>:</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Message Types</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>ERROR</TD>
 *     <TD>Message added to Action level error messages.  Default.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>FIELD</TD>
 *     <TD>Message added to field error messages using same name as field.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>MESSAGE</TD>
 *     <TD>Message added to Action level info messages.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>WARNING</TD>
 *     <TD>Message added to Action level warning messages.</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>Other parameters</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Other Parameters</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>message</TD>
 *     <TD>If not empty string, text message to display.  Used if <CODE>messageKey</CODE> is empty string or finds no 
 *         message.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>messageKey</TD>
 *     <TD>If not empty string, message key to retrieve from Action's TextProvider.  Uses <CODE>message</CODE> if no key 
 *         provided or no message found.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>param1</TD>
 *     <TD>Parameter recognised by custom converter.  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>param2</TD>
 *     <TD>Parameter recognised by custom converter.  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>param3</TD>
 *     <TD>Parameter recognised by custom converter.  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>param4</TD>
 *     <TD>Parameter recognised by custom converter.  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>param5</TD>
 *     <TD>Parameter recognised by custom converter.  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>parsedFieldName</TD>
 *     <TD>If not empty string, name of field to contain parsed value.  Defaults to 'parsed' plus capitalised field 
 *         name, such as 'parsedLimit'.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>processNoValue</TD><TD>If true, can convert empty string and format null or empty collection.  Defaults to 
 *                              false.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>validatorClass</TD>
 *     <TD>Class of client supplied validator.  See notes above.</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>When applied to a view helper field, {@link StrutsMiscellaneousLibrary#updateDisplay StrutsMiscellaneousLibrary#updateDisplay} 
 *    formats from a source record field with the same name but only certain combinations are recognised.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Formatting Annotation Usage</CAPTION>
 *   <TR>
 *     <TH>Display field type</TH>
 *     <TH>Record field type</TH>
 *     <TH>Notes</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String</TD>
 *     <TD>Collection of collection converter recipient type</TD>
 *     <TD></TD>
 *   </TR>
 * </TABLE>
 */
@Documented
@Inherited
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface CustomCollectionConversion {
    public String message() default "";
    public String messageKey() default "";
    public MessageType messageType() default MessageType.ERROR;
    public String param1() default "";
    public String param2() default "";
    public String param3() default "";
    public String param4() default "";
    public String param5() default "";
    public String parsedFieldName() default "";
    public boolean processNoValue() default false;
    public Class<? extends CollectionConverter<CustomCollectionConversion,?>> validatorClass();
}
