package name.matthewgreet.strutscommons.interceptor;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.action.ValidationAware2;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.policy.Adjuster;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionValidator;
import name.matthewgreet.strutscommons.policy.ConversionResult;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.policy.NonConversionValidator;
import name.matthewgreet.strutscommons.policy.PostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.PostConversionValidator;
import name.matthewgreet.strutscommons.policy.ValidationResult;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup.Configuration;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.ConfiguredPolicy;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.ModelDriven;
import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.interceptor.MethodFilterInterceptor;
import com.opensymphony.xwork2.interceptor.ValidationAware;

/**
 * <P>Struts 2 Interceptor for parsing annotated form fields from strings to another data type and writes error 
 * messages where data type conversion fails.  If the Action implements {@link ModelDriven}, the model is recognised as 
 * the form, not the action.  Struts Actions are expected to implement {@link ValidationAware} and, if message keys are 
 * used, implement {@link TextProvider}.</P>
 * 
 * <P>This differs from the standard ConversionErrorInterceptor by allowing writing to the general error list, not just 
 * field errors, and, if a message key is used, using the Action's text provider directly, not going through the Value 
 * Stack.  </P> 
 *
 * <P><u>Interceptor parameters:</u></P>
 * <DL>
 *  <DT>disabled</DT>
 *  <DD>If true, this interceptor is skipped.  Defaults to false.</DD>
 * </DL>
 *
 * <P><U>Extending the interceptor:</U></P>
 *
 * <P>Various points of the core algorithm can be overridden to customize behaviour.  See javadoc of protected methods.</P>
 *
 * <P> <u>Example code:</u></P>
 * <PRE>
 * &lt;action name="someAction" class="com.examples.SomeAction"&gt;
 *     &lt;interceptor-ref name="annotationValidation"/&gt;
 *     &lt;interceptor-ref name="validation"/&gt;
 *     &lt;result name="success"&gt;good_result.ftl&lt;/result&gt;
 * &lt;/action&gt;
 * <!-- END SNIPPET: example -->
 * </PRE>
 *
 * @deprecated Only recognises form fields in formatted/unformatted pairs (pair conversion mode).  Use 
 * {@link AnnotationValidationInterceptor2}, which recognises this and other modes.
 */
@Deprecated
public class AnnotationValidationInterceptor extends MethodFilterInterceptor {
	/**
	 * Describes a form field annotation, its type, and an instance of a policy it configures, if applicable. 
	 */
    public static class AnnotationEntry<T> {
        private Annotation annotation;
        private ConfiguredPolicy<T> configuredPolicy;
        
        
        public AnnotationEntry(Annotation annotation, ConfiguredPolicy<T> configuredPolicy) {
            super();
            this.annotation = annotation;
            this.configuredPolicy = configuredPolicy;
        }
        
        
        public Annotation getAnnotation() {
            return annotation;
        }
        public ConfiguredPolicy<T> getConfiguredPolicy() {
            return configuredPolicy;
        }
        
    }
    
    /**
     * Describes the result of a attempting to convert from a string to the generic type, and the field to receive it, 
     * successful or not.
     */
    public static class ConversionFieldResult<T> {
        private ConversionResult<T> conversionResult;
        private Field recipeintField;
        
        public ConversionFieldResult(Field recipeintField, ConversionResult<T> conversionResult) {
            super();
            this.recipeintField = recipeintField;
            this.conversionResult = conversionResult;
        }
        
        
        public ConversionResult<T> getConversionResult() {
            return conversionResult;
        }
        public Field getRecipeintField() {
            return recipeintField;
        }
    }
    
    private static final long serialVersionUID = 2689404505465349761L;
    
    /**
     * Returns all member fields of a class, whether directly declared by the class or inherited. 
     */
    public static Collection<Field> getProperties(Class<?> type) {
        Collection<Field> fields;
        Class<?> c;
        
        fields = new ArrayList<Field>();
        for (c = type; c != null; c = c.getSuperclass()) {
            fields.addAll(Arrays.asList(c.getDeclaredFields()));
        }
        return fields;
    }
    

    private Logger LOG = LogManager.getLogger(AnnotationValidationInterceptor.class);
    
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_CLASSES , required = false)
    private String acceptClasses = InterceptorCommonLibrary.DEFAULT_ACCEPT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_PACKAGES, required = false)
    private String acceptPackages = InterceptorCommonLibrary.DEFAULT_ACCEPT_PACKAGES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_CLASSPATH_SCANNING_REPLACE_BUILT_IN, required = false)
    private String classpathScanningReplaceBuiltIn = InterceptorCommonLibrary.DEFAULT_CLASSPATH_SCANNING_REPLACE_BUILT_IN;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ENABLE_CLASSPATH_SCANNING, required = false)
    private String enableClassScanning = InterceptorCommonLibrary.DEFAULT_ENABLE_CLASSPATH_SCANNING;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_CLASSES, required = false)
    private String rejectClasses = InterceptorCommonLibrary.DEFAULT_REJECT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_PACKAGES, required = false)
    private String rejectPackages = InterceptorCommonLibrary.DEFAULT_REJECT_PACKAGES;
    
    private boolean disabled;
    
    /**
     * Core algorithm for processing all annotations on a String form field, which can adjust, convert and validate it.
     *  
     * @param invocation Action processing state.
     * @param validationAware Message handling interface of Action, which can hypothetically be null.
     * @param textProvider Message localisation interface of Action, which can hypothetically be null.
     * @param allFormFields All form fields of the form, whether source or conversion target fields.
     * @param formField Field to be processed. 
     * @param form Form being processed.
     * @param modelDriven Whether Action being processed implements ModelDriven.
     */
    private <T> void processFormField(ActionInvocation invocation, ValidationAware validationAware, TextProvider textProvider,
            Collection<Field> allFormFields, Field formField, Object form, boolean modelDriven) throws Exception {
        Adjuster<?> adjuster;
        AnnotationEntry<T> collectionConversionAnnotationEntry, conversionAnnotationEntry;
        ConfiguredPolicy<T> configuredPolicy;
        CollectionConverter<?,T> collectionConverter;
        CollectionPostConversionValidator<?,T> collectionPostConversionValidator;
        ConversionFieldResult<T> conversionFieldResult;
        ConversionResult<T> conversionResult;
        Converter<?,T> converter;
        NonConversionValidator<?> nonConversionValidator;
        PostConversionValidator<?,T> postConversionValidator;
        ValidationResult validationResult;
        T parsedValue;
        Collection<T> parsedCollectionValue;
        Annotation[] annotations;
        Field recipientField;
        List<AnnotationEntry<T>> adjusterAnnotationEntries, nonConversionAnnotationEntries, postConversionAdjusterAnnotationEntries, postConversionValidatorAnnotationEntries;
        List<AnnotationEntry<T>> collectionPostConversionAdjusterAnnotationEntries, collectionPostConversionValidatorAnnotationEntries;
        String fieldName, fieldValue;
        boolean rejected, shortCircuit;

        formField.setAccessible(true);
        fieldName = formField.getName();
        fieldValue = (String)formField.get(form);
        if (fieldValue == null) {
            fieldValue = "";
            formField.set(form, fieldValue);
        }
        
        adjusterAnnotationEntries = new ArrayList<>();
        collectionPostConversionAdjusterAnnotationEntries = new ArrayList<>();
        collectionPostConversionValidatorAnnotationEntries = new ArrayList<>();
        nonConversionAnnotationEntries = new ArrayList<>();
        postConversionAdjusterAnnotationEntries = new ArrayList<>();
        postConversionValidatorAnnotationEntries = new ArrayList<>();
        conversionAnnotationEntry = null;
        collectionConversionAnnotationEntry = null;
        annotations = formField.getAnnotations();
        for (Annotation annotation: annotations) {
            try {
            	configuredPolicy = getConfiguredPolicy(formField, annotation);
            }
            catch (Exception e) {
            	configuredPolicy = ConfiguredPolicy.makeNAResult(annotation);
                LOG.error("Client supplied validator for form field failed " + 
                        "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                        "  field=" + formField.getName() + "  annotation=" + annotation.annotationType(), e);
            }
            switch (configuredPolicy.getAnnotationUsage()) {
            case ADJUSTER:
                adjusterAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            case COLLECTION_CONVERT:
                collectionConversionAnnotationEntry = new AnnotationEntry<T>(annotation, configuredPolicy);
                break;
            case COLLECTION_POST_ADJUSTER:
                collectionPostConversionAdjusterAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            case COLLECTION_POST_VALIDATION:
                collectionPostConversionValidatorAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            case CONVERT:
                conversionAnnotationEntry = new AnnotationEntry<T>(annotation, configuredPolicy);
                break;
			case MANUAL_PARAMETER_CONVERSION:
                // N/A
				break;
            case NA:
                // N/A
                break;
            case NON_CONVERT_VALIDATION:
                nonConversionAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            case POST_ADJUSTER:
                postConversionAdjusterAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            case POST_VALIDATION:
                postConversionValidatorAnnotationEntries.add(new AnnotationEntry<T>(annotation, configuredPolicy));
                break;
            }
        }
        // Adjusters
        for (AnnotationEntry<?> adjustmentAnnotationEntry: adjusterAnnotationEntries) {
            adjuster = adjustmentAnnotationEntry.getConfiguredPolicy().getAdjuster();
            fieldValue = adjust(form, formField, adjustmentAnnotationEntry.getAnnotation(), adjuster, fieldValue);
        }
        // Non-conversion validators
        rejected = false;
        for (AnnotationEntry<?> nonConvertAnnotationEntry: nonConversionAnnotationEntries) {
            nonConversionValidator = nonConvertAnnotationEntry.getConfiguredPolicy().getNonConversionValidator();
            validationResult = validateNonConversion(formField, nonConvertAnnotationEntry, fieldValue);
            if (!validationResult.getSuccess()) {
                rejected = true;
                shortCircuit = getNonConversionShortCircuit(formField, nonConvertAnnotationEntry, fieldValue);
            } else {
                shortCircuit = false;
            }
            checkNonConversionMessage(formField, nonConvertAnnotationEntry.getAnnotation(), nonConversionValidator,  
                    validationResult, validationAware, textProvider);
            if (shortCircuit) {
                break;
            }
        }
        if (!rejected && conversionAnnotationEntry != null) {
        	// Conversion (single value)
            conversionFieldResult = convert(invocation, form, allFormFields, formField, fieldName, fieldValue, conversionAnnotationEntry, 
                modelDriven);
            if (conversionFieldResult != null) {
                converter = conversionAnnotationEntry.getConfiguredPolicy().getConverter();
                recipientField = conversionFieldResult.getRecipeintField();
                conversionResult = conversionFieldResult.getConversionResult();
                parsedValue = conversionResult.getParsedValue();
                checkConversionMessage(formField, conversionAnnotationEntry.getAnnotation(), converter,  
                        conversionFieldResult, validationAware, textProvider);
                // Post-conversion adjusters (single value)
                for (AnnotationEntry<T> postConversionAdjusterAnnotationEntry: postConversionAdjusterAnnotationEntries) {
                    parsedValue = postConvertAdjust(invocation, form, formField, recipientField, conversionResult, 
                            parsedValue, postConversionAdjusterAnnotationEntry, modelDriven);
                }
                // Post-conversion validations (single value)
                for (AnnotationEntry<T> postConversionValidatorAnnotationEntry: postConversionValidatorAnnotationEntries) {
                    validationResult = postConvertValidate(invocation, form, formField, recipientField, 
                    		conversionResult, parsedValue, postConversionValidatorAnnotationEntry, modelDriven);
                    if (validationResult != null) {
                        postConversionValidator = postConversionValidatorAnnotationEntry.getConfiguredPolicy().getPostConversionValidator();
                        if (!validationResult.getSuccess()) {
                            rejected = true;
                            shortCircuit = getPostConversionShortCircuit(formField, postConversionValidatorAnnotationEntry, parsedValue);
                        } else {
                            shortCircuit = false;
                        }
                        checkPostConversionMessage(formField, postConversionValidatorAnnotationEntry.getAnnotation(), postConversionValidator,
                            conversionFieldResult, validationResult, validationAware, textProvider);
                    } else {
                        shortCircuit = false;   // Ignore misapplied post-conversion validator
                    }
                    if (shortCircuit) {
                        break;
                    }
                }
            } else {
                rejected = true;    // Abandon misapplied converter
            }
        } else if (!rejected && collectionConversionAnnotationEntry != null) {
        	// Conversion (collection)
            conversionFieldResult = collectionConvert(invocation, form, allFormFields, formField, fieldName, fieldValue, collectionConversionAnnotationEntry, 
                    modelDriven);
            if (conversionFieldResult != null) {
                collectionConverter = collectionConversionAnnotationEntry.getConfiguredPolicy().getCollectionConverter();
                recipientField = conversionFieldResult.getRecipeintField();
                conversionResult = conversionFieldResult.getConversionResult();
                parsedCollectionValue = conversionResult.getParsedCollectionValue();
                checkCollectionConversionMessage(formField, collectionConversionAnnotationEntry.getAnnotation(), collectionConverter,  
                    conversionFieldResult, validationAware, textProvider);
                // Post-conversion adjusters (collection)
                for (AnnotationEntry<T> collectionPostConversionAdjusterAnnotationEntry: collectionPostConversionAdjusterAnnotationEntries) {
                    parsedCollectionValue = collectionPostConvertAdjust(invocation, form, formField, recipientField, conversionResult, 
                            parsedCollectionValue, collectionPostConversionAdjusterAnnotationEntry, modelDriven);
                }
            	// Post-conversion validations (collection)
                for (AnnotationEntry<T> collectionPostConversionAnnotationEntry: collectionPostConversionValidatorAnnotationEntries) {
                    validationResult = collectionPostConvertValidate(invocation, form, formField, recipientField, 
                    		conversionResult, parsedCollectionValue, collectionPostConversionAnnotationEntry, modelDriven);
                    if (validationResult != null) {
                        collectionPostConversionValidator = collectionPostConversionAnnotationEntry.getConfiguredPolicy().getCollectionPostConversionValidator();
                        if (!validationResult.getSuccess()) {
                            rejected = true;
                            shortCircuit = getCollectionPostConversionShortCircuit(formField, collectionPostConversionAnnotationEntry, parsedCollectionValue);
                        } else {
                            shortCircuit = false;
                        }
                        checkCollectionPostConversionMessage(formField, collectionPostConversionAnnotationEntry.getAnnotation(), 
                            collectionPostConversionValidator, conversionFieldResult, validationResult, validationAware, textProvider);
                    } else {
                        shortCircuit = false;   // Ignore misapplied post-conversion validator
                    }
                    if (shortCircuit) {
                        break;
                    }
                }
            } else {
                rejected = true;
            }
        }
    }
    
    /**
     * @throws IllegalAccessException 
     * @throws IllegalArgumentException 
     * 
     */
    protected String adjust(Object form, Field formField, Annotation annotation, Adjuster<?> adjuster, String fieldValue) throws Exception {
        if (fieldValue.length() > 0 || adjuster.getProcessNoValue()) {
	        fieldValue = adjuster.adjust(fieldValue);
	        formField.set(form, fieldValue);
        }
        return fieldValue;
    }
    
    /**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     */
    protected <T> void checkCollectionConversionMessage(Field unconvertedField, Annotation annotation, CollectionConverter<?,T> collectionConverter, 
            ConversionFieldResult<T> conversionFieldResult, ValidationAware validationAware, TextProvider textProvider) {
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        if (!conversionFieldResult.getConversionResult().getSuccess()) {
            if (conversionFieldResult.getConversionResult().getMessage() != null) {
                message = conversionFieldResult.getConversionResult().getMessage();
            } else {
                message = collectionConverter.getMessage();
            }
            if (conversionFieldResult.getConversionResult().getMessageKey() != null) {
                messageKey = conversionFieldResult.getConversionResult().getMessageKey();
            } else {
                messageKey = collectionConverter.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (conversionFieldResult.getConversionResult().getMessageType() != null) {
                messageType = conversionFieldResult.getConversionResult().getMessageType();
            } else {
                messageType = collectionConverter.getMessageType();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, unconvertedField.getName(), finalMessage);
                break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Returns whether converting validator can convert form field value to the recipient field's type. 
     */
    protected boolean checkCollectionConversionRecipientDataType(Field unconvertedField, Annotation annotation, Field recipientField, 
            Class<?> recipientClass) {
        return InterceptorCommonLibrary.checkCollectionRecipientDataType(recipientField, recipientClass);
    }
    
    /**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     */
    protected <T> void checkCollectionPostConversionMessage(Field unconvertedField, Annotation annotation,
            CollectionPostConversionValidator<?,T> collectionPostConversionValidator, ConversionFieldResult<T> conversionFieldResult,
            ValidationResult validationResult, ValidationAware validationAware, TextProvider textProvider) {
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        if (!validationResult.getSuccess()) {
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = collectionPostConversionValidator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = collectionPostConversionValidator.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = collectionPostConversionValidator.getMessageType();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, unconvertedField.getName(), finalMessage);
                break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Returns whether post conversion validator can use the recipient field's type. 
     */
    protected boolean checkCollectionPostConversionRecipientDataType(Annotation annotation, Field recipientField, 
            Class<?> recipientClass) {
        return InterceptorCommonLibrary.checkCollectionRecipientDataType(recipientField, recipientClass);
    }
    
    /**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     */
    protected <T> void checkConversionMessage(Field unconvertedField, Annotation annotation, Converter<?,T> converter, 
            ConversionFieldResult<T> conversionFieldResult, ValidationAware validationAware, TextProvider textProvider) {
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        if (!conversionFieldResult.getConversionResult().getSuccess()) {
            if (conversionFieldResult.getConversionResult().getMessage() != null) {
                message = conversionFieldResult.getConversionResult().getMessage();
            } else {
                message = converter.getMessage();
            }
            if (conversionFieldResult.getConversionResult().getMessageKey() != null) {
                messageKey = conversionFieldResult.getConversionResult().getMessageKey();
            } else {
                messageKey = converter.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (conversionFieldResult.getConversionResult().getMessageType() != null) {
                messageType = conversionFieldResult.getConversionResult().getMessageType();
            } else {
                messageType = converter.getMessageType();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, unconvertedField.getName(), finalMessage);
                break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Returns whether converting validator can convert form field value to the recipient field's type. 
     */
    protected boolean checkConversionRecipientDataType(Field unconvertedField, Annotation annotation, Field recipientField, 
            Class<?> recipientClass) {
        return InterceptorCommonLibrary.checkRecipientDataType(recipientField, recipientClass);
    }
    
    /**
     * Checks if non-conversion validation results means a message must be written and writes appropriate type if needed.  
     */
    protected void checkNonConversionMessage(Field unconvertedField, Annotation annotation, NonConversionValidator<?> validator, 
            ValidationResult validationResult, ValidationAware validationAware, TextProvider textProvider) {
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        if (!validationResult.getSuccess()) {
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = validator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = validator.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = validator.getMessageType();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, unconvertedField.getName(), finalMessage);
                break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Returns whether post conversion validator can use the recipient field's type. 
     */
    protected boolean checkPostConversionRecipientDataType(Annotation annotation, Field recipientField, 
            Class<?> recipientClass) {
        return InterceptorCommonLibrary.checkRecipientDataType(recipientField, recipientClass);
    }
    
    /**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     */
    protected <T> void checkPostConversionMessage(Field unconvertedField, Annotation annotation, PostConversionValidator<?,T> postConversionValidator, 
            ConversionFieldResult<T> conversionFieldResult, ValidationResult validationResult, ValidationAware validationAware, TextProvider textProvider) {
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        if (!validationResult.getSuccess()) {
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = postConversionValidator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = postConversionValidator.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = postConversionValidator.getMessageType();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, unconvertedField.getName(), finalMessage);
                break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Converts form field value and returns result, or null if recipient field missing or compatibility checks fail. 
     */
    protected <T> ConversionFieldResult<T> collectionConvert(ActionInvocation invocation, Object form, Collection<Field> allFormFields, Field formField, 
            String fieldName, String fieldValue, AnnotationEntry<T> collectionConversionAnnotationEntry, boolean modelDriven) {
        Annotation colectionConversionAnnotation;
        ConversionFieldResult<T> result;
        CollectionConverter<?,T> collectionConverter;
        ConversionResult<T> validationResult;
        Class<T> recipientFieldClass;
        Field recipientField;
        Collection<T> parsedValue;
        String recipientFieldName;
        boolean compatible;

        colectionConversionAnnotation = collectionConversionAnnotationEntry.getAnnotation();
        collectionConverter = collectionConversionAnnotationEntry.getConfiguredPolicy().getCollectionConverter();
        recipientFieldName = collectionConverter.getRecipientFieldName();
        if (recipientFieldName == null || recipientFieldName.length() == 0) {
            recipientFieldName = getDefaultRecipientName(formField, fieldName);
        }
        recipientField = getRecipientField(formField, colectionConversionAnnotation, allFormFields, recipientFieldName);
        if (recipientField == null) {
            LOG.error("Form field annotation refers to non-existent recipient field, explicitly or by default" + 
                "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                "  field=" + formField.getName() + "  annotation=" + colectionConversionAnnotation.annotationType());
            return null;
        }
        compatible = checkCollectionConversionRecipientDataType(formField, colectionConversionAnnotation, recipientField, collectionConverter.getRecipientClass());
        if (!compatible) {
            LOG.error("Recipient field cannot receive converted value" + 
                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                    "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
                    "  annotation=" + colectionConversionAnnotation.annotationType() + 
                    "  allowed classes=" + collectionConverter.getRecipientClass());
            return null;
        }
        if (fieldValue.length() > 0 || collectionConverter.getProcessNoValue()) {
	        parsedValue = null;
	        try {
	            recipientFieldClass = InterceptorCommonLibrary.getTypeFromCollectionField(recipientField);
	            validationResult = validateCollectionConversion(formField, colectionConversionAnnotation, collectionConverter, fieldValue, 
	                    recipientField, recipientFieldClass);
	            parsedValue = validationResult.getParsedCollectionValue();
	            if (validationResult.getSuccess() && parsedValue != null) {
	                recipientField.setAccessible(true);
	                recipientField.set(form, parsedValue);
	            }
	        }
	        catch (Exception e) {
	            LOG.error("Setting recipient field failed" + 
	                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
	                    "  field=" + formField.getName() + "  annotation=" + colectionConversionAnnotation.annotationType() + 
	                    "  value to set=" + parsedValue, e);
	            validationResult = ConversionResult.makeFailureResult();
	        }
	        result = new ConversionFieldResult<T>(recipientField, validationResult);
	        return result;
        } else {
        	return new ConversionFieldResult<T>(recipientField, ConversionResult.makeSkippedResult());
        }
    }
    
    /**
     * Returns converted value adjusted by post conversion adjuster.
     */
    protected <T> Collection<T> collectionPostConvertAdjust(ActionInvocation invocation, Object form, Field formField,
            Field recipientField, ConversionResult<T> conversionResult, Collection<T> parsedCollectionValue,
            AnnotationEntry<T> collectionPostConversionAdjusterAnnotationEntry, boolean modelDriven) throws Exception {
        Annotation postConversionAnnotation;
        CollectionPostConversionAdjuster<?,T> collectionPostConversionAdjuster;
        Collection<T> result;
        boolean compatible;
        
        postConversionAnnotation = collectionPostConversionAdjusterAnnotationEntry.getAnnotation();
        collectionPostConversionAdjuster = collectionPostConversionAdjusterAnnotationEntry.getConfiguredPolicy().getCollectionPostConversionAdjuster();
        
        result = parsedCollectionValue;
        if ((conversionResult.getSuccess() && parsedCollectionValue != null) || collectionPostConversionAdjuster.getProcessNoValue()) {
            compatible = checkPostConversionRecipientDataType(postConversionAnnotation, recipientField, 
                    collectionPostConversionAdjuster.getRecipientClass());
            if (!compatible) {
                LOG.warn("Collection post conversion adjuster has incompatible data type for recipient field" + 
                        "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                        "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
                        "  annotation=" + postConversionAnnotation.annotationType() + 
                        "  allowed classes=" + collectionPostConversionAdjuster.getRecipientClass());
                result = parsedCollectionValue;
            }
            result = collectionPostConversionAdjuster.adjust(parsedCollectionValue);
        }
        return result;
    }

    /**
     * Return validation result of converted form field value, or null if compatibility checks fail. 
     */
    protected <T> ValidationResult collectionPostConvertValidate(ActionInvocation invocation, Object form, Field formField, Field recipientField, 
    		ConversionResult<T> conversionResult, Collection<T> parsedValue, AnnotationEntry<T> postConversionAnnotationEntry, boolean modelDriven) throws Exception {
        Annotation postConversionAnnotation;
        ValidationResult result;
        CollectionPostConversionValidator<?,T> collectionPostConversionValidator;
        Class<T> recipientFieldClass;
        boolean compatible;
        
        postConversionAnnotation = postConversionAnnotationEntry.getAnnotation();
        collectionPostConversionValidator = postConversionAnnotationEntry.getConfiguredPolicy().getCollectionPostConversionValidator();

        if ((conversionResult.getSuccess() && parsedValue != null) || collectionPostConversionValidator.getProcessNoValue()) {
	        compatible = checkCollectionPostConversionRecipientDataType(postConversionAnnotation, recipientField, 
	                collectionPostConversionValidator.getRecipientClass());
	        if (!compatible) {
	            LOG.error("Post conversion validator has incompatible data type for recipient field" + 
	                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
	                    "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
	                    "  annotation=" + postConversionAnnotation.annotationType() + 
	                    "  allowed class=" + collectionPostConversionValidator.getRecipientClass());
	            return null;
	        }
	        
	        recipientFieldClass = InterceptorCommonLibrary.getTypeFromCollectionField(recipientField);
	        result = validateCollectionPostConversion(formField, postConversionAnnotationEntry.getAnnotation(), 
	                postConversionAnnotationEntry.getConfiguredPolicy().getCollectionPostConversionValidator(),
	                parsedValue, recipientField, recipientFieldClass);
	        return result;
        } else {
        	return ValidationResult.makeSuccessResult();
        }
    }
    
    /**
     * Converts form field value and returns result, or null if recipient field missing or compatibility checks fail. 
     */
    @SuppressWarnings("unchecked")
    protected <T> ConversionFieldResult<T> convert(ActionInvocation invocation, Object form, Collection<Field> allFormFields, Field formField, 
            String fieldName, String fieldValue, AnnotationEntry<T> conversionAnnotationEntry, boolean modelDriven) {
        Annotation conversionAnnotation;
        ConversionFieldResult<T> result;
        Converter<?,T> converter;
        ConversionResult<T> conversionResult;
        Field recipientField;
        T parsedValue;
        String recipientFieldName;
        boolean compatible;

        conversionAnnotation = conversionAnnotationEntry.getAnnotation();
        converter = conversionAnnotationEntry.getConfiguredPolicy().getConverter();
        recipientFieldName = converter.getRecipientFieldName();
        if (recipientFieldName == null || recipientFieldName.length() == 0) {
            recipientFieldName = getDefaultRecipientName(formField, fieldName);
        }
        recipientField = getRecipientField(formField, conversionAnnotation, allFormFields, recipientFieldName);
        if (recipientField == null) {
            LOG.error("Form field annotation refers to non-existent recipient field, explicitly or by default" + 
                "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                "  field=" + formField.getName() + "  annotation=" + conversionAnnotation.annotationType());
            return null;
        }
        compatible = checkConversionRecipientDataType(formField, conversionAnnotation, recipientField, converter.getRecipientClass());
        if (!compatible) {
            LOG.error("Recipient field cannot receive converted value" + 
                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                    "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
                    "  annotation=" + conversionAnnotation.annotationType() + 
                    "  allowed classes=" + converter.getRecipientClass());
            return null;
        }
        if (fieldValue.length() > 0 || converter.getProcessNoValue()) {
	        parsedValue = null;
	        try {
	        	conversionResult = validateConversion(formField, conversionAnnotation, converter, fieldValue, 
	                    recipientField, (Class<T>)recipientField.getType());
	            parsedValue = conversionResult.getParsedValue();
	            if (conversionResult.getSuccess() && parsedValue != null) {
	                recipientField.setAccessible(true);
	                recipientField.set(form, parsedValue);
	            }
	        }
	        catch (Exception e) {
	            LOG.error("Setting recipient field failed" + 
	                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
	                    "  field=" + formField.getName() + "  annotation=" + conversionAnnotation.annotationType() + 
	                    "  value to set=" + parsedValue, e);
	            conversionResult = ConversionResult.makeFailureResult();
	        }
	        result = new ConversionFieldResult<T>(recipientField, conversionResult);
	        return result;
        } else {
        	return new ConversionFieldResult<T>(recipientField, ConversionResult.makeSkippedResult());
        }
    }
    
    @Override
    protected String doIntercept(ActionInvocation invocation) throws Exception {
        TextProvider textProvider;
        ValidationAware validationAware;
        Collection<Field> allFormFields;
        Object action, form;
        boolean modelDriven;

        if (!disabled) {
	        action = invocation.getAction();
	        modelDriven = action instanceof ModelDriven;
	        if (modelDriven) {
	            form = ((ModelDriven<?>)action).getModel();
	        } else {
	            form = action;
	        }
	        if (action instanceof TextProvider) {
	            textProvider = (TextProvider)action;
	        } else {
	            textProvider = null;
	        }
	        if (action instanceof ValidationAware) {
	            validationAware = (ValidationAware)action;
	        } else {
	            validationAware = null;
	        }
	        
	        allFormFields = getProperties(form.getClass());
	        for (Field formField: allFormFields) {
	            if (formField.getType().isAssignableFrom(String.class)) {
	                processFormField(invocation, validationAware, textProvider, allFormFields, formField, form, modelDriven);
	            }
	        }
        }
        
        return invocation.invoke();
    }
    
    /**
     * Returns validator (and formatter) that processes an annotated form field, or result of NA type if not recognised. 
     */
    protected <T> ConfiguredPolicy<T> getConfiguredPolicy(Field unconvertedField, Annotation annotation) throws Exception {
		Configuration configuration;
		DefaultPolicyLookup defaultPolicyLookup;
		
		configuration = new Configuration();
		configuration.setAcceptClasses(acceptClasses);
		configuration.setAcceptPackages(acceptPackages);
		configuration.setClasspathScanningReplaceBuiltIn(Boolean.parseBoolean(classpathScanningReplaceBuiltIn));
		configuration.setEnableClasspathScanning(Boolean.parseBoolean(enableClassScanning));
		configuration.setRejectClasses(rejectClasses);
		configuration.setRejectPackages(rejectPackages);
		defaultPolicyLookup = DefaultPolicyLookup.getInstance(configuration);
        return InterceptorCommonLibrary.getConfiguredPolicy(annotation, defaultPolicyLookup);
    }

    /**
     * Returns whether a rejection by the non-conversion validator should stop further validation. 
     */
    protected <T> boolean getCollectionPostConversionShortCircuit(Field unconvertedField, AnnotationEntry<?> annotationEntry, Collection<T> parsedValue) {
        return annotationEntry.getConfiguredPolicy().getCollectionPostConversionValidator().getShortCircuit();
    }

    /**
     * Returns the default field name to receive a converted form field.  
     */
    protected String getDefaultRecipientName(Field unconvertedField, String fieldName) {
        return InterceptorCommonLibrary.getDefaultRecipientName(fieldName);
    }

    /**
     * Returns message to use, extracted from text provider if required.  textProvider can be null if Action does not 
     * implement it.
     */
    protected String getMessage(TextProvider textProvider, String messageKey, String message) {
        String result;
        
        if (textProvider != null && messageKey != null && messageKey.length() > 0 && textProvider.hasKey(messageKey)) {
            result = textProvider.getText(messageKey);
        } else {
            result = message;
        }
        return result;
    }

    /**
     * Returns whether a rejection by the non-conversion validator should stop further validation. 
     */
    protected boolean getNonConversionShortCircuit(Field unconvertedField, AnnotationEntry<?> annotationEntry, String formValue) {
        return annotationEntry.getConfiguredPolicy().getNonConversionValidator().getShortCircuit();
    }

    /**
     * Returns whether a rejection by the non-conversion validator should stop further validation. 
     */
    protected <T> boolean getPostConversionShortCircuit(Field unconvertedField, AnnotationEntry<?> annotationEntry, T parsedValue) {
        return annotationEntry.getConfiguredPolicy().getPostConversionValidator().getShortCircuit();
    }

    /**
     * Finds field to receive converted value from fields found in the form, or null if not found. 
     */
    protected Field getRecipientField(Field unconvertedField, Annotation annotation, Collection<Field> fields, String recipientFieldName) {
        Field result;
        
        result = null;
        for (Field field: fields) {
            if (field.getName().equals(recipientFieldName)) {
                result = field;
                break;
            }
        }
        return result;
    }

    /**
     * Returns an empty collection to set the recipient field and will accept converted values. 
     */
    protected <T> Collection<T> makeCollectionForRecipient(Class<?> recipientFieldClass) {
        return InterceptorCommonLibrary.makeCollectionForRecipient(recipientFieldClass);
    }

    /**
     * Returns converted value adjusted by post conversion adjuster.
     */
    protected <T> T postConvertAdjust(ActionInvocation invocation, Object form, Field formField, Field recipientField, 
            ConversionResult<T> conversionResult, T parsedValue, AnnotationEntry<T> postConversionAnnotationEntry, boolean modelDriven) throws Exception {
        Annotation postConversionAnnotation;
        PostConversionAdjuster<?,T> postConversionAdjuster;
        T result;
        boolean compatible;
        
        postConversionAnnotation = postConversionAnnotationEntry.getAnnotation();
        postConversionAdjuster = postConversionAnnotationEntry.getConfiguredPolicy().getPostConversionAdjuster();
        
        result = parsedValue;
        if ((conversionResult.getSuccess() && parsedValue != null) || postConversionAdjuster.getProcessNoValue()) {
            compatible = checkPostConversionRecipientDataType(postConversionAnnotation, recipientField, 
                    postConversionAdjuster.getRecipientClass());
            if (!compatible) {
                LOG.warn("Post conversion adjuster has incompatible data type for recipient field" + 
                        "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
                        "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
                        "  annotation=" + postConversionAnnotation.annotationType() + 
                        "  allowed classes=" + postConversionAdjuster.getRecipientClass());
                result = parsedValue;
            }
            result = postConversionAdjuster.adjust(parsedValue);
        }
        return result;
    }

    /**
     * Return validation result of converted form field value, or null if compatibility checks fail. 
     */
    @SuppressWarnings("unchecked")
    protected <T> ValidationResult postConvertValidate(ActionInvocation invocation, Object form, Field formField, Field recipientField, 
    		ConversionResult<T> conversionResult, T parsedValue, AnnotationEntry<T> postConversionAnnotationEntry, boolean modelDriven) throws Exception {
        Annotation postConversionAnnotation;
        PostConversionValidator<?,T> postConversionValidator;
        ValidationResult result;
        boolean compatible;
        
        postConversionAnnotation = postConversionAnnotationEntry.getAnnotation();
        postConversionValidator = postConversionAnnotationEntry.getConfiguredPolicy().getPostConversionValidator();

        if ((conversionResult.getSuccess() && parsedValue != null) || postConversionValidator.getProcessNoValue()) {
	        compatible = checkPostConversionRecipientDataType(postConversionAnnotation, recipientField, 
	                postConversionValidator.getRecipientClass());
	        if (!compatible) {
	            LOG.error("Post conversion validator has incompatible data type for recipient field" + 
	                    "  Struts action=" + invocation.getAction().getClass() + "  model driven=" + modelDriven + 
	                    "  field=" + formField.getName() + "  recipient field=" + recipientField.getName() + 
	                    "  annotation=" + postConversionAnnotation.annotationType() + 
	                    "  allowed classes=" + postConversionValidator.getRecipientClass());
	            return null;
	        }
	        
	        result = validatePostConversion(formField, postConversionAnnotationEntry.getAnnotation(), 
	            postConversionAnnotationEntry.getConfiguredPolicy().getPostConversionValidator(),
	            parsedValue, recipientField, (Class<T>)recipientField.getType());
	        return result;
        } else {
        	return ValidationResult.makeSuccessResult();
        }
    }

    /**
     * Calls validator to convert string form field value to recipient field of expected data type. 
     */
    protected <T> ConversionResult<T> validateCollectionConversion(Field unconvertedField, Annotation annotation, 
            CollectionConverter<?,T> collectionConverter, String formValue, Field recipientField, Class<T> recipientClass) throws Exception {
        if (formValue != null || collectionConverter.getProcessNoValue()) {
            return collectionConverter.convert(formValue, recipientField.getType(), recipientClass);
        } else {
            return ConversionResult.makeSkippedCollectionResult(makeCollectionForRecipient(recipientField.getType()));
        }
    }

    /**
     * Calls validator to check a converted form field. 
     */
    protected <T> ValidationResult validateCollectionPostConversion(Field unconvertedField, Annotation annotation, 
            CollectionPostConversionValidator<?,T> collectionPostConversionValidator, Collection<T> convertedValue, Field recipientField, Class<?> recipientClass) throws Exception {
        return collectionPostConversionValidator.validate(convertedValue);
    }

    /**
     * Calls validator to convert string form field value to recipient field of expected data type. 
     */
    protected <T> ConversionResult<T> validateConversion(Field unconvertedField, Annotation annotation, Converter<?,T> converter,
            String fieldValue, Field recipientField, Class<T> recipientClass) throws Exception {
        if (fieldValue != null || converter.getProcessNoValue()) {
            return converter.convert(fieldValue, recipientClass);
        } else {
            return ConversionResult.makeSkippedResult();
        }
    }

    /**
     * Calls validator to check a form field before any conversion. 
     */
    protected ValidationResult validateNonConversion(Field unconvertedField, AnnotationEntry<?> annotationEntry, String fieldValue) throws Exception {
        if (fieldValue.length() > 0 || annotationEntry.getConfiguredPolicy().getNonConversionValidator().getProcessNoValue()) {
        	return annotationEntry.getConfiguredPolicy().getNonConversionValidator().validate(fieldValue);
        } else {
        	return ValidationResult.makeSuccessResult();
        }
    }

    /**
     * Calls validator to check a converted form field. 
     */
    protected <T> ValidationResult validatePostConversion(Field unconvertedField, Annotation annotation, 
            PostConversionValidator<?,T> postConversionValidator, T convertedValue, Field recipientField, Class<T> recipientClass) throws Exception {
        return postConversionValidator.validate(convertedValue);
    }

    /**
     * Writes message to Action's field errors for field.  ValidationAware can be null if Action does not implement it.  
     */
    protected void writeFieldError(ValidationAware validationAware, String fieldName, String message) {
        if (validationAware != null) {
            validationAware.addFieldError(fieldName, message);;
        }
    }

    /**
     * Writes message to Action's action errors.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeGeneralError(ValidationAware validationAware, String message) {
        if (validationAware != null) {
            validationAware.addActionError(message);
        }
    }

    /**
     * Writes message to Action's action messages.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeInfoMessage(ValidationAware validationAware, String message) {
        if (validationAware != null) {
            validationAware.addActionMessage(message);
        }
    }
    
    /**
     * Writes message to Action's action warnings.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeWarningMessage(ValidationAware validationAware, String message) {
        if (validationAware != null && validationAware instanceof ValidationAware2) {
            ((ValidationAware2)validationAware).addActionWarning(message);
        } else if (validationAware != null) {
            validationAware.addActionMessage(message);
        }
    }
    
	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	

}
