package name.matthewgreet.strutscommons.interceptor;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.form.FormattableForm;
import name.matthewgreet.strutscommons.util.DefaultFormFormatter;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup;
import name.matthewgreet.strutscommons.util.FormFormatter;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary;
import name.matthewgreet.strutscommons.util.PolicyLookup;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup.Configuration;

import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;
import com.opensymphony.xwork2.interceptor.PreResultListener;

/**
 * <P>Struts 2 Interceptor used by view Actions for writing formatted versions of forms onto the Value Stack, so they're 
 * displayed instead of the unformatted form, using any conversion annotations on form fields.  Such forms must 
 * implement {@link FormattableForm}.  Does not apply to injected forms as its conversion errors are copied.</P>
 * 
 * <P><U>Interceptor parameters:</U></P>
 * <DL>
 *   <DT>disabled</DT>
 *   <DD>If true, all processing for this action is disabled.  Defaults to false.</DD>
 * </DL>
 *
 * <P><U>Extending the interceptor:</U></P>
 * <P>The following methods can be overridden :-</P>
 * <DL>
 *   <DT>makeFormFormatLibrary</DT><DD>Creates helper that formats form.</DD>
 *   <DT>makePolicyLookup</DT><DD>Creates helper that finds policy implementation from the annotation that configures it.</DD>
 * </DL>
 *
 * <P> <u>Example code:</u></P>
 * <PRE>
 * &lt;action name="someAction" class="com.examples.SomeAction"&gt;
 *     &lt;interceptor-ref name="annotationValidation"/&gt;
 *     &lt;interceptor-ref name="validation"/&gt;
 *     &lt;interceptor-ref name="formFormatter"/&gt;
 *     &lt;result name="success"&gt;good_result.ftl&lt;/result&gt;
 * &lt;/action&gt;
 * <!-- END SNIPPET: example -->
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class FormFormatterInterceptor extends AbstractInterceptor {
	public class FormFormatterPreResultListener implements PreResultListener {

		public void beforeResult(ActionInvocation invocation, String resultCode) {
			FormFormatter formFormatLibrary;
			PolicyLookup policyLookup;
			
			policyLookup = makePolicyLookup();
			formFormatLibrary = makeFormFormatLibrary();
			formFormatLibrary.formatForms(policyLookup);
		}

	}
	
    private static final long serialVersionUID = 2689404505465349761L;
    
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_CLASSES , required = false)
    private String acceptClasses = InterceptorCommonLibrary.DEFAULT_ACCEPT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_PACKAGES, required = false)
    private String acceptPackages = InterceptorCommonLibrary.DEFAULT_ACCEPT_PACKAGES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_CLASSPATH_SCANNING_REPLACE_BUILT_IN, required = false)
    private String classpathScanningReplaceBuiltIn = InterceptorCommonLibrary.DEFAULT_CLASSPATH_SCANNING_REPLACE_BUILT_IN;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ENABLE_CLASSPATH_SCANNING, required = false)
    private String enableClassScanning = InterceptorCommonLibrary.DEFAULT_ENABLE_CLASSPATH_SCANNING;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_CLASSES, required = false)
    private String rejectClasses = InterceptorCommonLibrary.DEFAULT_REJECT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_PACKAGES, required = false)
    private String rejectPackages = InterceptorCommonLibrary.DEFAULT_REJECT_PACKAGES;
    
    private boolean disabled = false;

    
	@SuppressWarnings("unused")
	private Logger LOG = LogManager.getLogger(FormFormatterInterceptor.class);
    

    /**
     * Creates and returns helper that formats form field values. 
     */
	protected FormFormatter makeFormFormatLibrary() {
		return new DefaultFormFormatter();
	}
	
	/**
	 * Creates and returns helper that finds a policy implementation from a form field adjuster, converter, or validator 
	 * annotation.  Does not scan for non-built-in, non-custom policies if disabledClassScanner is true.
	 */
	protected PolicyLookup makePolicyLookup() {
		Configuration configuration;
		
		configuration = new Configuration();
		configuration.setAcceptClasses(acceptClasses);
		configuration.setAcceptPackages(acceptPackages);
		configuration.setClasspathScanningReplaceBuiltIn(Boolean.parseBoolean(classpathScanningReplaceBuiltIn));
		configuration.setEnableClasspathScanning(Boolean.parseBoolean(enableClassScanning));
		configuration.setRejectClasses(rejectClasses);
		configuration.setRejectPackages(rejectPackages);
		return DefaultPolicyLookup.getInstance(configuration);
	}

	
	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	

	@Override
	public String intercept(ActionInvocation invocation) throws Exception {
        if (!disabled) {
        	invocation.addPreResultListener(new FormFormatterPreResultListener());
        }
        return invocation.invoke();
	}

	

}
