package name.matthewgreet.strutscommons.util;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionContext;

import name.matthewgreet.strutscommons.annotation.BooleanConversion;
import name.matthewgreet.strutscommons.annotation.EnumConversion;
import name.matthewgreet.strutscommons.annotation.Form;
import name.matthewgreet.strutscommons.annotation.IntegerConversion;
import name.matthewgreet.strutscommons.interceptor.FormFormatterInterceptor;
import name.matthewgreet.strutscommons.policy.AbstractCustomCollectionFormatterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCustomFormatterSupport;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationUsage;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.ConfiguredPolicy;

/**
 * Various static library functions for use by library clients not covered other libraries.
 */
@SuppressWarnings("deprecation")
public class StrutsMiscellaneousLibrary {
    public static class FormattedFormsResult implements Serializable {
		private static final long serialVersionUID = 5652867827375400916L;
		
		Map<String,Map<String,String>> singleValueFormFields = new HashMap<>();
    	Map<String,Map<String,List<String>>> multipleValueFormFields = new HashMap<>();
    	
		public List<String> getMultipleValueFormField(String formName, String fieldName) {
			Map<String,List<String>> fakeForm;
			List<String> result;
			
			fakeForm = multipleValueFormFields.get(formName);
			if (fakeForm != null) {
				result = fakeForm.get(fieldName);
				if (result == null) {
					result = new ArrayList<>();
				}
			} else {
				result = new ArrayList<>();
			}
			return result;

		}
		public void setMultipleValueFormField(String formName, String fieldName, List<String> formattedValues) {
			Map<String,List<String>> fakeForm;
			
			fakeForm = multipleValueFormFields.get(formName);
			if (fakeForm == null) {
				fakeForm = new HashMap<>();
				multipleValueFormFields.put(formName, fakeForm);
			}
			fakeForm.put(fieldName, formattedValues);
		}
		
		public String getSingleValueFormField(String formName, String fieldName) {
			Map<String,String> fakeForm;
			String result;
			
			fakeForm = singleValueFormFields.get(formName);
			if (fakeForm != null) {
				result = fakeForm.get(fieldName);
				if (result == null) {
					result = "";
				}
			} else {
				result = "";
			}
			return result;
		}
		public void setSingleValueFormField(String formName, String fieldName, String formattedValue) {
			Map<String,String> fakeForm;
			
			fakeForm = singleValueFormFields.get(formName);
			if (fakeForm == null) {
				fakeForm = new HashMap<>();
				singleValueFormFields.put(formName, fakeForm);
			}
			fakeForm.put(fieldName, formattedValue);
		}
		
    }
    
    private static final Logger LOG = LogManager.getLogger(StrutsMiscellaneousLibrary.class);
    
    /**
     * Formats the annotated form fields from the associated, unformatted halves of every formatted/unformatted field 
     * pair.  See {@link BooleanConversion}, {@link EnumConversion}, {@link IntegerConversion} etc.
     * 
     * <P>This is a legacy function and is called by view Struts Actions that don't use the 
     * {@link FormFormatterInterceptor} interceptor.</P>
     */
    public static <T> void formatForm(Object form) {
        Annotation[] annotations;
        ConfiguredPolicy<T> configuredPolicy;
        Collection<Field> formFields;
        
        if (form != null) {
            formFields = InterceptorCommonLibrary.getProperties(form.getClass());
            for (Field formField: formFields) {
                if (String.class.isAssignableFrom(formField.getType())) {
                    try {
                        annotations = formField.getAnnotations();
                        for (Annotation annotation: annotations) {
                            try {
                            	configuredPolicy = InterceptorCommonLibrary.getConfiguredPolicy(annotation, DefaultPolicyLookup.getInstance());
                                if (configuredPolicy.getAnnotationUsage() == AnnotationUsage.CONVERT) {
                                	InterceptorCommonLibrary.formatFormField(form, formFields, formField, configuredPolicy);
                                    break;
                                } else if (configuredPolicy.getAnnotationUsage() == AnnotationUsage.COLLECTION_CONVERT) {
                                	InterceptorCommonLibrary.formatCollectionFormField(form, formFields, formField, configuredPolicy);
                                    break;
                                }
                            }
                            catch (Exception e) {
                                LOG.error("Client supplied validator for form field failed " + 
                                        "  Struts action=" + ActionContext.getContext().getActionName() +  
                                        "  field=" + formField.getName() + "  annotation=" + annotation.annotationType(), e);
                            }
                        }
                    }
                    catch (Exception e) {
    
                        LOG.error("Security violation when accessing form field" + 
                                "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                                "  field=" + formField.getName(), e);
                        return;
                    }
                }
            }
        }
    }
    
    /**
     * <P>For all the current Action's forms, formats the annotated, formatted halves of every formatted/unformatted 
     * field pair from the unformatted halves, except for any form that was retrieved and injected (usually a rejected 
     * form).  See {@link BooleanConversion}, {@link EnumConversion}, {@link IntegerConversion} etc.</P>
     * 
     * <P>This is a legacy function and is called by view Struts Actions that don't use the 
     *    {@link FormFormatterInterceptor} interceptor.</P>
     */
    public static void formatForms() {
    	Form formAnnotation;
        ActionContext actionContext;
        Collection<Field> actionFields;
        Object action;
        boolean doForm;

        actionContext = ActionContext.getContext();
        action = actionContext.getActionInvocation().getAction();
        actionFields = InterceptorCommonLibrary.getProperties(action.getClass());
        for (Field actionField: actionFields) {
            doForm = !InterceptorCommonLibrary.fieldReceivedStoredForm(actionField) && !InterceptorCommonLibrary.fieldIsNotForm(actionField);
            if (doForm) {
                try {
                    actionField.setAccessible(true);
                    formAnnotation = actionField.getAnnotation(Form.class);
                    if (formAnnotation == null || !formAnnotation.disableFormatting()) {
                    	formatForm(actionField.get(action));
                    }
                }
                catch (Exception e) {
                    LOG.error("Security violation when accessing form" + 
                            "  Struts action=" + action.getClass() + "  field=" + actionField.getName(), e);
                    return;
                }
            }
        }
    }
    
    /**
     * Returns the formatted forms created by {@link FormFormatterInterceptor}.  As these are only created by a Struts 2 
     * pre-result listener set by the Interceptor, no result is available till then.
     */
    @SuppressWarnings("unchecked")
	public static synchronized FormattedFormsResult getFormattedForms() {
    	FormattedFormsResult result;
    	Map<String,Map<String,Object>> fakeForms;
    	Map<String,Object> fakeForm;
    	Object fakeField;
    	String formName, fieldName;
    	
    	fakeForms = InterceptorCommonLibrary.getActionContextFormattedForms();
    	result = new FormattedFormsResult();
    	for (Entry<String, Map<String, Object>> fakeFormEntry: fakeForms.entrySet()) {
    		formName = fakeFormEntry.getKey();
    		fakeForm = fakeFormEntry.getValue();
    		for (Entry<String, Object> fakeFieldEntry: fakeForm.entrySet()) {
    			fieldName = fakeFieldEntry.getKey();
    			fakeField = fakeFieldEntry.getValue();
    			if (List.class.isAssignableFrom(fakeField.getClass())) {
    				result.setMultipleValueFormField(formName, fieldName, (List<String>)fakeField);
    			} else if (String.class.isAssignableFrom(fakeField.getClass())) {
    				result.setSingleValueFormField(formName, fieldName, (String)fakeField);
    			}
    		}
    	}
    	return result;
    }

    /**
     * <P>Sets string or string collection fields of <CODE>display</CODE> from fields of <CODE>record</CODE> with the 
     * same name but formatted according to the conversion annotations, such as {@link IntegerConversion}, on 
     * <CODE>display</CODE> fields, or default converter if no annotation.  
     * If <CODE>record</CODE> is null, recipient <CODE>display</CODE> fields are set according to the converter's 
     * formatting of null values (usually empty string).  This function is useful for displaying database records.</P>
     * 
     * <P>For each display field, the exact behaviour depends on it, the matching record field, and conversion 
     * annotation.</P>
     * <TABLE CLASS="main">
     * <THEAD>
     *  <TR>
     *    <TH>&nbsp;</TH>
     *    <TH COLSPAN="3">Display field</TH>
     *  </TR>
     *  <TR>
     *    <TH STYLE="width: 100px;">Source field</TH>
     *    <TH STYLE="width: 70px;">Single value</TH>
     *    <TH STYLE="width: 70px;">Array</TH>
     *    <TH STYLE="width: 70px;">Collection</TH>
     *  </TR>
     * </THEAD>
     * <TBODY>
     *  <TR CLASS="row_odd">
     *    <TD STYLE="font-weight: bold;">Single value</TD>
     *    <TD>Yes</TD>
     *    <TD></TD>
     *    <TD></TD>
     *  </TR>
     *  <TR CLASS="row_even">
     *    <TD STYLE="font-weight: bold;">Array</TD>
     *    <TD></TD>
     *    <TD>Yes</TD>
     *    <TD>Yes</TD>
     *  </TR>
     *  <TR CLASS="row_odd">
     *    <TD STYLE="font-weight: bold;">Collection</TD>
     *    <TD>Yes<SUP>1</SUP></TD>
     *    <TD>Yes</TD>
     *    <TD>Yes</TD>
     *  </TR>
     * </TBODY>
     *  <CAPTION>Display formatting modes</CAPTION>
     * </TABLE>
     * <P><SUP>1</SUP> Uses collection converter instead.</P>
     * 
     * <P>Miscellaneous notes.</P>
     * <UL>
     *   <LI>Source field type must apply to converter's type.</LI>
     *   <LI>Custom converters only for formatting are best derived from {@link AbstractCustomFormatterSupport} or 
     *       {@link AbstractCustomCollectionFormatterSupport}.</LI>
     *   <LI>The default converters for booleans and enumerations don't format to user friendly text, so custom 
     *       converters are better.</LI>
     *   <LI>Does not follow associations.</LI>
     * </UL> 
     */
	public static void updateDisplay(Class<?> recordClass, Object record, Object display) {
		DefaultDisplayFormatter displayFormatter;
		
		displayFormatter = new DefaultDisplayFormatter();
		displayFormatter.updateDisplay(recordClass, record, display);
	}

    /**
     * <P>Same as {@link #updateDisplay #updateDisplay} but can use a different set of formatters.</P>
     */ 
	public static void updateDisplayWithPolicyLookup(Class<?> recordClass, Object record, Object display, PolicyLookup policyLookup) {
		DefaultDisplayFormatter displayFormatter;
		
		displayFormatter = new DefaultDisplayFormatter();
		displayFormatter.setPolicyLookup(policyLookup);
		displayFormatter.updateDisplay(recordClass, record, display);
	}

}
