package name.matthewgreet.strutscommons.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import name.matthewgreet.strutscommons.action.LoggingAware;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.util.StrutsMiscellaneousLibrary;

/**
 * <P>Defines a form field (whether on a Struts Action or model of ModelDriven Struts Action) should be parsed to 
 * a Date type and error message to display if this fails.  By default, DD/MM/YYYY format is expected 
 * but this can be overridden or use user's locale by setting format to empty string.  Also, defines a view helper field 
 * receives a formatted datetime value.  This is the default converter for Date fields.</P>
 *
 * <P>There are two modes of behaviour for conversion, depending on the type of form field.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Validation Annotation Usage</CAPTION>
 *   <TR>
 *     <TH>Form field type</TH>
 *     <TH>Mode</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String</TD>
 *     <TD>Pair conversion</TD>
 *     <TD>Converts string field value to Date field named by parsedFieldName</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Date</TD>
 *     <TD>Auto conversion</TD>
 *     <TD>Converts directly from request parameter</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>By default, messages are written as Action error messages for form processing Actions and logged as warnings for 
 *    viewer Actions but this can be overridden by <CODE>messageType</CODE>.  For viewer Actions, it's recommended they 
 *    implement {@link LoggingAware} and set a logging message type.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Message Types</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>DEFAULT</TD>
 *     <TD>Uses default setting of controlling interceptor or library, which is usually Action error messages for form 
 *         processing Actions and logged as warnings for view ActionsMessage.  Default.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>ERROR</TD>
 *     <TD>Message added to Action level error messages.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>FIELD</TD>
 *     <TD>Message added to field error messages using same name as field.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>IGNORE</TD>
 *     <TD>Message is not shown to user or logged.  This is not recommended.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>LOG_DEBUG</TD>
 *     <TD>Logged to Action's Log4j2 logger (or controlling interceptor's if not available) at DEBUG level.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>LOG_ERROR</TD>
 *     <TD>Logged to Action's Log4j2 logger (or controlling interceptor's if not available) at ERROR level.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>LOG_INFO</TD>
 *     <TD>Logged to Action's Log4j2 logger (or controlling interceptor's if not available) at INFO level.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>LOG_TRACE</TD>
 *     <TD>Logged to Action's Log4j2 logger (or controlling interceptor's if not available) at TRACE level.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>LOG_WARN</TD>
 *     <TD>Logged to Action's Log4j2 logger (or controlling interceptor's if not available) at WARN level.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>MESSAGE</TD>
 *     <TD>Message added to Action level info messages.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>WARNING</TD>
 *     <TD>Message added to Action level warning messages.</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>Other parameters</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Other Parameters</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>actionContextTimeZone</TD>
 *     <TD>Name of ActionContext value of TimeZone instance for time formatting.  Resorts to system time zone if no 
 *         value found or name is empty string.  Ignored if timezone format is not empty string (which defaults to 
 *         "UTC").  Defaults to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>format</TD>
 *     <TD>If not empty string, fixed date format, using SimpleDateFormat format.  Defaults to dd/MM/yyyy.  Set to empty 
 *         string to use format according to locale.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>localeCountry</TD>
 *     <TD>If not empty string, country code for specific locale, as used in Locale.  If set, localeLanguage must also 
 *         be set.  Ignored if format is not empty string.  Set to empty string to use ActionContext's locale.  Defaults 
 *         to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>localeLanguage</TD>
 *     <TD>If not empty string, language code for specific locale, as used in Locale.  If set, localeCountry must also 
 *         be set.  Ignored if format is not empty string.  Set to empty string to use ActionContext's locale.  Defaults 
 *         to empty string.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>message</TD>
 *     <TD>If not empty string, text message to display.  Used if <CODE>messageKey</CODE> is empty string or finds no 
 *         message.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>messageKey</TD>
 *     <TD>If not empty string, message key to retrieve from Action's TextProvider.  Uses <CODE>message</CODE> if no key 
 *         provided or no message found.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>parsedFieldName</TD>
 *     <TD>If not empty string, name of field to contain parsed value.  Defaults to 'parsed' plus capitalised field 
 *         name, such as 'parsedLimit'.  Only applies for pair conversion.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>style</TD>
 *     <TD>Date style of a locale's date format to be used, same as DateFormat date style.  Ignored if format is not 
 *         empty string.  Defaults to SHORT.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>timeStyle</TD>
 *     <TD>Time style of a locale's time format to be used, same as DateFormat date style, unless set to NONE.  Ignored 
 *         if format is not empty string.  Defaults to NONE.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>timezone</TD>
 *     <TD>If not empty string, name of time zone recognised by TimeZone.getTimeZone().  Set to empty string to default 
 *         to time zone found by actionContextTimeZone.  Defaults to "UTC".</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>When applied to a view helper field, {@link StrutsMiscellaneousLibrary#updateDisplay StrutsMiscellaneousLibrary#updateDisplay} 
 *   formats from a source record field with the same name but only certain combinations are recognised.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Formatting Annotation Usage</CAPTION>
 *   <TR>
 *     <TH>Display field type</TH>
 *     <TH>Record field type</TH>
 *     <TH>Notes</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String</TD>
 *     <TD>Date</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>String array</TD>
 *     <TD>Date array</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String array</TD>
 *     <TD>Date collection</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>String collection</TD>
 *     <TD>Date array</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String collection</TD>
 *     <TD>Date collection</TD>
 *     <TD></TD>
 *   </TR>
 * </TABLE>
 */
@Documented
@Inherited
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface DateConversion {
    public enum Style {FULL, LONG, MEDIUM, SHORT}
    public enum TimeStyle {FULL, LONG, MEDIUM, SHORT, NONE}
    
    public String actionContextTimeZone() default "";
    public String format() default "dd/MM/yyyy";
    public String localeCountry() default "";
    public String localeLanguage() default "";
    public String message() default "";
    public String messageKey() default "";
    public MessageType messageType() default MessageType.DEFAULT;
    public String parsedFieldName() default "";
    public Style style() default Style.SHORT;
    public TimeStyle timeStyle() default TimeStyle.NONE;
    public String timezone() default "UTC";
}
