package name.matthewgreet.strutscommons.interceptor;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.form.Form;
import name.matthewgreet.strutscommons.util.AnnotationValidatior;
import name.matthewgreet.strutscommons.util.DefaultAnnotationValidator;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup;
import name.matthewgreet.strutscommons.util.DefaultPolicyLookup.Configuration;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary;
import name.matthewgreet.strutscommons.util.PolicyLookup;

import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.ModelDriven;
import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.interceptor.MethodFilterInterceptor;
import com.opensymphony.xwork2.interceptor.ValidationAware;

/**
 * <P>Struts 2 Interceptor for setting form fields from request parameters, adjusting, converting, validating, and 
 * writing error messages where data type conversion fails, according to form field annotations.  If the Action 
 * implements {@link ModelDriven}, the model is recognised as the form, not the action.  Struts Actions are expected to 
 * implement {@link ValidationAware} and, if message keys are used, implement {@link TextProvider}.</P>
 * 
 * <P>Annotations are applied to a form's member variables.  The standard set are found in 
 * {@link name.matthewgreet.strutscommons.annotation} and includes customisable ones.  Annotations configure various 
 * kinds of policies and typically don't apply for missing form field values (@Required being a notable exception).</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Types of Annotations</CAPTION>
 *   <TR>
 *     <TH>Policy</TH>
 *     <TH>Example</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Adjuster</TD>
 *     <TD>@ToUpperCase</TD>
 *     <TD>Modifies parameter value before validation or conversion.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Non-conversion validator</TD>
 *     <TD>@MaxLength</TD>
 *     <TD>Validates parameter value before any conversion.  Usually only applicable to string fields, with @Required as 
 *         a notable exception.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Converter</TD>
 *     <TD>@DateConversion</TD>
 *     <TD>Converts string request parameter and sets form field.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Post-conversion adjuster</TD>
 *     <TD>@ToStartOfDay</TD>
 *     <TD>Modifies converted form field value.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Post-conversion validator</TD>
 *     <TD>@IntegerRange</TD>
 *     <TD>Validates converted form field value.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Collection converter</TD>
 *     <TD>@IntegerCSVConversion</TD>
 *     <TD>Splits and converts string request parameter and sets collection form field.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Collection post-conversion adjuster</TD>
 *     <TD>None</TD>
 *     <TD>Modifies converted values of collection form field.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Collection post-conversion validator</TD>
 *     <TD>@RequiredIntegerEntries</TD>
 *     <TD>Validates converted values of collection form field.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Manual parameter conversion</TD>
 *     <TD>@ManualParameterConversion</TD>
 *     <TD>A special annotation indicating form field is processed in manual parameter conversion mode (see below).</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>Each form field is processed in one of the modes described below.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Validation Modes</CAPTION>
 *   <TR>
 *     <TH>Mode</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Auto conversion</TD>
 *     <TD>Directly sets form field if it passes non-conversion validation and conversion.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Default conversion</TD>
 *     <TD>Same as auto conversion but a default converter is used for the field's data type.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Manual parameter conversion</TD>
 *     <TD>Adjustment, conversion, and validation is done by form code.  Form must implement {@link Form}.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Pair conversion</TD>
 *     <TD>Sets a string form field and applies adjusters and validators and converts to a paired, non-string form 
 *         field.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>No conversion</TD>
 *     <TD>Sets a string form field and applies adjusters and validators.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Set only</TD>
 *     <TD>Sets a string form field and no policies are applied.</TD>
 *   </TR>
 * </TABLE>
 * <P>Except where a @ManualParameterConversion annotation is used, how each mode applies is summarised below.</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Mode Usage</CAPTION>
 *   <TR>
 *     <TH>Field type</TH>
 *     <TH>Converter annotation</TH>
 *     <TH>Mode</TH>
 *     <TH>Notes</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String</TD>
 *     <TD>No</TD>
 *     <TD>No conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>String</TD>
 *     <TD>Yes</TD>
 *     <TD>Pair conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Non-string</TD>
 *     <TD>No</TD>
 *     <TD>Default conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Non-string</TD>
 *     <TD>Yes</TD>
 *     <TD>Auto conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String array</TD>
 *     <TD>&nbsp;</TD>
 *     <TD>Set only</TD>
 *     <TD>Converters don't apply to arrays</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>String collection</TD>
 *     <TD>No</TD>
 *     <TD>Set only</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>String collection</TD>
 *     <TD>Yes</TD>
 *     <TD>Auto conversion</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Non-string array</TD>
 *     <TD>&nbsp;</TD>
 *     <TD>Manual parameter conversion</TD>
 *     <TD>Converters don't apply to arrays</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Non-string collection</TD>
 *     <TD>No</TD>
 *     <TD>Default conversion</TD>
 *     <TD>Few default converters for collection types exist</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Non-string collection</TD>
 *     <TD>Yes</TD>
 *     <TD>Auto conversion</TD>
 *     <TD></TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>This replaces the standard AnnotationValidationInterceptor, ConversionErrorInterceptor and ParametersInterceptor 
 * and differs in various aspects.</P>
 * <UL>
 *   <LI>Unlike ParametersInterceptor, directly sets form fields, rather than setting properties on the Value Stack.</LI>
 *   <LI>Unlike ParametersInterceptor, does not convert multiple request parameters with the same name.  See above.</LI>
 *   <LI>Unlike ConversionErrorInterceptor, error messages aren't restricted to field errors.</LI>
 * </UL>
 * 
 * <P><U>Interceptor parameters:</U></P>
 *
 * <DL>
 *  <DT>defaultMessageType</DT>
 *  <DD>Where to write conversion and validation failures where an annotation uses {@link MessageType} of DEFAULT.  
 *      Cannot be DEFAULT.  Defaults to ERROR.</DD>
 *  <DT>disabled</DT>
 *  <DD>If true, this interceptor is skipped.  Defaults to false.</DD>
 *  <DT>ignoreNonAnnotatedFields</DT>
 *  <DD>If true, does not process form fields with no recognised annotations.  This is used for viewer Actions where 
 *      some member fields are used for display, not accept form parameters.  Defaults to false.</DD>
 *  <DT>paramNameMaxLength</DT>
 *  <DD>Ignores all request parameters with names longer than this.  Defaults to 100.</DD>
 * </DL>
 *
 * <P><U>Extending the interceptor:</U></P>
 * <DL>
 *  <DT>makeAnnotationValidationLibrary</DT><DD>Creates helper that adjust, validates and converts form field values.</DD>
 *  <DT>makePolicyLookup</DT><DD>Creates helper that returns the policy implementation for a form field annotation.</DD>
 * </DL>
 * <P> <u>Example code:</u></P>
 * <PRE>
 * &lt;action name="someAction" class="com.examples.SomeAction"&gt;
 *     &lt;interceptor-ref name="annotationValidation"/&gt;
 *     &lt;interceptor-ref name="validation"/&gt;
 *     &lt;result name="success"&gt;good_result.ftl&lt;/result&gt;
 * &lt;/action&gt;
 * <!-- END SNIPPET: example -->
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class AnnotationValidationInterceptor2 extends MethodFilterInterceptor {
    private static final long serialVersionUID = 2689404505465349761L;
    
    public static final int PARAM_NAME_MAX_LENGTH = 100;
    
    @SuppressWarnings("unused")
	private Logger LOG = LogManager.getLogger(AnnotationValidationInterceptor2.class);
    
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_CLASSES , required = false)
    private String acceptClasses = InterceptorCommonLibrary.DEFAULT_ACCEPT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ACCEPT_PACKAGES, required = false)
    private String acceptPackages = InterceptorCommonLibrary.DEFAULT_ACCEPT_PACKAGES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_CLASSPATH_SCANNING_REPLACE_BUILT_IN, required = false)
    private String classpathScanningReplaceBuiltIn = InterceptorCommonLibrary.DEFAULT_CLASSPATH_SCANNING_REPLACE_BUILT_IN;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_ENABLE_CLASSPATH_SCANNING, required = false)
    private String enableClassScanning = InterceptorCommonLibrary.DEFAULT_ENABLE_CLASSPATH_SCANNING;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_CLASSES, required = false)
    private String rejectClasses = InterceptorCommonLibrary.DEFAULT_REJECT_CLASSES;
    @Inject(value = InterceptorCommonLibrary.STRUTS_CONSTANT_REJECT_PACKAGES, required = false)
    private String rejectPackages = InterceptorCommonLibrary.DEFAULT_REJECT_PACKAGES;
    
    private MessageType defaultMessageType = MessageType.ERROR;
    private boolean disabled;
    private boolean ignoreNonAnnotatedFields = false;
    private int paramNameMaxLength = PARAM_NAME_MAX_LENGTH;
    

    @Override
    protected String doIntercept(ActionInvocation invocation) throws Exception {
    	AnnotationValidatior annotationValidationLibrary;
    	
        if (!disabled) {
        	annotationValidationLibrary = makeAnnotationValidationLibrary();
        	annotationValidationLibrary.validate();
        }
        
        return invocation.invoke();
    }

    /**
     * Creates and returns helper that validations form fields according to their annotations. 
     */
	protected AnnotationValidatior makeAnnotationValidationLibrary() {
		DefaultAnnotationValidator result;
		PolicyLookup policyLookup;
		
		policyLookup = makePolicyLookup();
		result = new DefaultAnnotationValidator();
		result.setDefaultMessageType(getDefaultMessageType());
		result.setIgnoreNonAnnotatedFields(getIgnoreNonAnnotatedFields());
		result.setParamNameMaxLength(getParamNameMaxLength());
		result.setPolicyLookup(policyLookup);
		return result;
	}

	/**
	 * Creates and returns helper that finds a policy implementation from a form field adjuster, converter, or validator 
	 * annotation.  Does not scan for non-built-in, non-custom policies if disabledClassScanner is true.
	 */
	protected PolicyLookup makePolicyLookup() {
		Configuration configuration;
		
		configuration = new Configuration();
		configuration.setAcceptClasses(acceptClasses);
		configuration.setAcceptPackages(acceptPackages);
		configuration.setClasspathScanningReplaceBuiltIn(Boolean.parseBoolean(classpathScanningReplaceBuiltIn));
		configuration.setEnableClasspathScanning(Boolean.parseBoolean(enableClassScanning));
		configuration.setRejectClasses(rejectClasses);
		configuration.setRejectPackages(rejectPackages);
		return DefaultPolicyLookup.getInstance(configuration);
	}

    
	public MessageType getDefaultMessageType() {
		return defaultMessageType;
	}
	public void setDefaultMessageType(MessageType defaultMessageType) {
		if (defaultMessageType != null && defaultMessageType != MessageType.DEFAULT) {
			this.defaultMessageType = defaultMessageType;
		}
	}

	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	
	public boolean getIgnoreNonAnnotatedFields() {
		return ignoreNonAnnotatedFields;
	}
	public void setIgnoreNonAnnotatedFields(boolean ignoreNonAnnotatedFields) {
		this.ignoreNonAnnotatedFields = ignoreNonAnnotatedFields;
	}

	public int getParamNameMaxLength() {
		return paramNameMaxLength;
	}
	public void setParamNameMaxLength(int paramNameMaxLength) {
		this.paramNameMaxLength = paramNameMaxLength;
	}


}
