package name.matthewgreet.strutscommons.interceptor;


import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts2.ServletActionContext;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.conversion.impl.ConversionData;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;

import name.matthewgreet.strutscommons.annotation.Form;
import name.matthewgreet.strutscommons.annotation.Form.Reception;
import name.matthewgreet.strutscommons.interceptor.FormStoreInterceptor.StoredForm;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary;


/**
 * <P>Retrieves a form stored in the session by {@link FormStoreInterceptor} and injects it into the action's member 
 * variable configured by the {@link Form} annotation to accept it.  It is usually part of the ViewStack interceptor
 * stack so a view action can display rejected form data.</P>
 * 
 * 
 * <P><U>Interceptor parameters</U></P>
 * <DL>
 * <DT>disabled</DT><DD>If true, all processing is disabled.  This is useful for standalone popup windows, especially 
 *                      self-refreshing ones, that never display messages.  Defaults to false.</DD>
 * </DL>
 * 
 * <P><U>Extending the interceptor</U></P>
 * <P>The following method could be overridden :-</P>
 * <DL>
 * <DT>fieldReceives</DT><DD>Whether Struts action member variable can be set to the stored form.</DD>
 * </DL>
 * 
 * <P><U>Example code</U></P>
 * <PRE>
 * &#064;InterceptorRefs({
 *   &#064;InterceptorRef(value="formRetrieve"),
 *   &#064;InterceptorRef(value="basicStack")
 *})
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class FormRetrieveInterceptor extends AbstractInterceptor {
    private static final long serialVersionUID = -6513506646328167140L;

    private Logger LOG = LogManager.getLogger(FormRetrieveInterceptor.class);
    

    private boolean disabled;


    public FormRetrieveInterceptor() {
        // Empty
    }
    
    /**
     * Returns all member fields an instance of a class has. 
     */
    private static Collection<Field> getProperties(Class<?> type) {
        Collection<Field> fields;
        Class<?> c;
        
        fields = new ArrayList<Field>();
        for (c = type; c != null; c = c.getSuperclass()) {
            fields.addAll(Arrays.asList(c.getDeclaredFields()));
        }
        return fields;
    }
    
    /**
     * Returns URL of current request, including query parameters.   
     */
    private String getFullURL() {
        if (ServletActionContext.getRequest().getQueryString() != null) {
            return ServletActionContext.getRequest().getRequestURL().append('?').append(ServletActionContext.getRequest().getQueryString()).toString();
        } else {
            return ServletActionContext.getRequest().getRequestURL().toString();
        }
    }

    /**
     * Adds stored conversion errors. 
     */
    protected void addConversionErrors(ActionInvocation invocation, StoredForm storedForm, Field field) throws Exception {
    	String fieldName, propertyName;
    	
    	fieldName = field.getName();
    	for (Entry<String, ConversionData> entry: storedForm.getConversionErrors().entrySet()) {
    		propertyName = fieldName + "." + entry.getKey();
    		invocation.getInvocationContext().getConversionErrors().put(propertyName, entry.getValue());
    	}
    }
    
    /**
     * Deletes stored form as unused.
     */
    protected void deleteStoredForm() {
        ActionContext.getContext().getSession().remove(FormStoreInterceptor.SESSION_STORED_FORM);
    }

    /**
     * Returns whether Struts action member variable can be set to the stored form, which is whether the member 
     * variable's type is compatible with the form and {@link Form#reception()} annotation settings accept form 
     * validation status. 
     */
    protected boolean fieldReceives(Class<?> actionClass, Field field, StoredForm storedForm) {
        Form formAnnotation;
        Reception reception;
        Class<?>[] processors;
        boolean result;

        if (!field.getType().isAssignableFrom(storedForm.getForm().getClass())) {
            return false;
        }
        
        if (field.getAnnotation(Form.class) != null) {
            formAnnotation = field.getAnnotation(Form.class);
        	reception = formAnnotation.reception();
        	processors = formAnnotation.processors();
        } else {
        	reception = Reception.ERROR;
        	processors = new Class<?>[0];
        }
        result = false;
        switch (reception) {
        case NEVER:   result = false; break;
        case ERROR:   result = storedForm.getInvalid(); break;
        case SUCCESS: result = !storedForm.getInvalid(); break;
        case ALWAYS:  result = true; break;
        }
        
        if (result && processors.length > 0) {
        	result = false;
        	for (Class<?> processor: processors) {
        		if (processor.isAssignableFrom(storedForm.getProcessor())) {
        			result = true;
        			break;
        		}
        	}
        }
        
        return result;
    }

    /**
     * Injects form into specific field. 
     */
    protected void injectForm(ActionInvocation invocation, StoredForm storedForm, Field field) throws Exception {
        field.setAccessible(true);
        field.set(invocation.getAction(), storedForm.getForm());
    }
    
    /**
     * Injects stored form into receiving field of current action and pushes conversion errors onto the Value Stack. 
     */
    protected void restoreForm(ActionInvocation invocation, StoredForm storedForm) {
        Collection<Field> fields;
        Set<String> fieldNames;
        Class<?> actionClass;
        boolean receive;
        
        actionClass = invocation.getAction().getClass();
        fields = getProperties(actionClass);
        fieldNames = new HashSet<>();
        for (Field field: fields) {
            receive = fieldReceives(actionClass, field, storedForm);
            if (receive) {
                try {
                    injectForm(invocation, storedForm, field);
                    addConversionErrors(invocation, storedForm, field);
                    fieldNames.add(field.getName());
                }
                catch (Exception e) {
                    LOG.error("Set stored form of Struts action member variable failed" + "  Struts action=" + 
                        invocation.getAction().getClass() + "  field=" + field.getName(), e);
                }
            }
        }
        
        storedForm.setOwningURL(getFullURL());
        storedForm.setOwningActionClass(actionClass);
        storedForm.setOwningActionFieldNames(fieldNames);
    }

    /**
     * Returns whether Action will not retrieve forms. 
     */
    public boolean getDisabled() {
        return disabled;
    }
    public void setDisabled(boolean value) {
        disabled = value;
    }
    
    
    @Override
    public String intercept(ActionInvocation invocation) throws Exception {
        StoredForm storedForm;
        
        if (!disabled) {
            storedForm = InterceptorCommonLibrary.getStoredForm();
            if (storedForm != null) {
                if (storedForm.getOwningURL() == null || storedForm.getOwningURL().equals(getFullURL())) {
                    restoreForm(invocation, storedForm);
                } else {
                    // Remove as user is displaying another page
                    deleteStoredForm();
                }
            }
        }
        return invocation.invoke();
    }

}
