package name.matthewgreet.strutscommons.util;

import java.io.File;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TransferQueue;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ModelDriven;
import com.opensymphony.xwork2.conversion.impl.ConversionData;

import name.matthewgreet.strutscommons.annotation.BigDecimalConversion;
import name.matthewgreet.strutscommons.annotation.BooleanConversion;
import name.matthewgreet.strutscommons.annotation.ByteConversion;
import name.matthewgreet.strutscommons.annotation.CharacterConversion;
import name.matthewgreet.strutscommons.annotation.CustomAdjuster;
import name.matthewgreet.strutscommons.annotation.CustomCollectionConversion;
import name.matthewgreet.strutscommons.annotation.CustomCollectionPostConversionAdjuster;
import name.matthewgreet.strutscommons.annotation.CustomCollectionPostConversionValidation;
import name.matthewgreet.strutscommons.annotation.CustomConversion;
import name.matthewgreet.strutscommons.annotation.CustomPostConversionAdjuster;
import name.matthewgreet.strutscommons.annotation.CustomPostConversionValidation;
import name.matthewgreet.strutscommons.annotation.CustomValidation;
import name.matthewgreet.strutscommons.annotation.DateConversion;
import name.matthewgreet.strutscommons.annotation.DoubleConversion;
import name.matthewgreet.strutscommons.annotation.EnumConversion;
import name.matthewgreet.strutscommons.annotation.FloatConversion;
import name.matthewgreet.strutscommons.annotation.FormField;
import name.matthewgreet.strutscommons.annotation.IntegerCSVConversion;
import name.matthewgreet.strutscommons.annotation.IntegerConversion;
import name.matthewgreet.strutscommons.annotation.IntegerRange;
import name.matthewgreet.strutscommons.annotation.LongConversion;
import name.matthewgreet.strutscommons.annotation.ToLowerCase;
import name.matthewgreet.strutscommons.annotation.ToStartOfDay;
import name.matthewgreet.strutscommons.annotation.ManualParameterConversion;
import name.matthewgreet.strutscommons.annotation.MaxLength;
import name.matthewgreet.strutscommons.annotation.MinInteger;
import name.matthewgreet.strutscommons.annotation.Regex;
import name.matthewgreet.strutscommons.annotation.Required;
import name.matthewgreet.strutscommons.annotation.RequiredIntegerEntries;
import name.matthewgreet.strutscommons.annotation.ShortConversion;
import name.matthewgreet.strutscommons.annotation.StringCSVConversion;
import name.matthewgreet.strutscommons.annotation.ToEndOfDay;
import name.matthewgreet.strutscommons.annotation.Trim;
import name.matthewgreet.strutscommons.exception.FormFieldAnnotationTypeMismatchException;
import name.matthewgreet.strutscommons.annotation.ToUpperCase;
import name.matthewgreet.strutscommons.interceptor.FormFormatterInterceptor;
import name.matthewgreet.strutscommons.interceptor.FormStoreInterceptor;
import name.matthewgreet.strutscommons.interceptor.FormStoreInterceptor.StoredForm;
import name.matthewgreet.strutscommons.policy.AbstractAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCollectionConverterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCollectionPostConversionAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractConverterSupport;
import name.matthewgreet.strutscommons.policy.AbstractNonConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.AbstractPostConversionAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractPostConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.Adjuster;
import name.matthewgreet.strutscommons.policy.BigDecimalConverter;
import name.matthewgreet.strutscommons.policy.BooleanConverter;
import name.matthewgreet.strutscommons.policy.ByteConverter;
import name.matthewgreet.strutscommons.policy.CharacterConverter;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionValidator;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.policy.DateConverter;
import name.matthewgreet.strutscommons.policy.DoubleConverter;
import name.matthewgreet.strutscommons.policy.EnumConverter;
import name.matthewgreet.strutscommons.policy.FloatConverter;
import name.matthewgreet.strutscommons.policy.IntegerCSVConverter;
import name.matthewgreet.strutscommons.policy.IntegerConverter;
import name.matthewgreet.strutscommons.policy.IntegerRangeValidator;
import name.matthewgreet.strutscommons.policy.LongConverter;
import name.matthewgreet.strutscommons.policy.MaxLengthValidator;
import name.matthewgreet.strutscommons.policy.MinIntegerValidator;
import name.matthewgreet.strutscommons.policy.NonConversionValidator;
import name.matthewgreet.strutscommons.policy.Policy;
import name.matthewgreet.strutscommons.policy.PostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.PostConversionValidator;
import name.matthewgreet.strutscommons.policy.RegexValidator;
import name.matthewgreet.strutscommons.policy.RequiredIntegerEntriesValidator;
import name.matthewgreet.strutscommons.policy.RequiredValidator;
import name.matthewgreet.strutscommons.policy.ShortConverter;
import name.matthewgreet.strutscommons.policy.StringCSVConverter;
import name.matthewgreet.strutscommons.policy.ToEndOfDayAdjuster;
import name.matthewgreet.strutscommons.policy.ToLowerCaseAdjuster;
import name.matthewgreet.strutscommons.policy.ToStartOfDayAdjuster;
import name.matthewgreet.strutscommons.policy.ToUpperCaseAdjuster;
import name.matthewgreet.strutscommons.policy.TrimAdjuster;

/**
 * Static functions used by validation Interceptors, form formating Interceptors and validators. 
 */
@SuppressWarnings("deprecation")
public class InterceptorCommonLibrary {
    /**
	 * Summarises the list of recognised annotations of a form field.
	 */
	public static class AnnotationEntries<T> {
		private List<ConfiguredPolicy<T>> adjusters;
		private ConfiguredPolicy<T> collectionConverter;
        private List<ConfiguredPolicy<T>> collectionPostConversionAdjusters;
		private List<ConfiguredPolicy<T>> collectionPostConversionValidators;
		private ConfiguredPolicy<T> converter;
		private boolean formField;
		private boolean manualParameterConversion;
        private List<ConfiguredPolicy<T>> postConversionAdjusters;
		private List<ConfiguredPolicy<T>> postConversionValidators;
		private List<ConfiguredPolicy<T>> validators;
		
		
		public AnnotationEntries() {
			adjusters = new ArrayList<>();
			formField = false;
			collectionConverter = null;
	        collectionPostConversionAdjusters = new ArrayList<>();
			collectionPostConversionValidators = new ArrayList<>();
			converter = null;
			manualParameterConversion = false;
            postConversionAdjusters = new ArrayList<>();
			postConversionValidators = new ArrayList<>();
			validators = new ArrayList<>();
		}
		
		public void addAnnotationUsageResult(ConfiguredPolicy<T> annotationUsageResult) {
			switch (annotationUsageResult.getAnnotationUsage()) {
			case ADJUSTER:
				adjusters.add(annotationUsageResult);
				break;
			case COLLECTION_CONVERT:
				collectionConverter = annotationUsageResult;
				break;
            case COLLECTION_POST_ADJUSTER:
                collectionPostConversionAdjusters.add(annotationUsageResult);
                break;
			case COLLECTION_POST_VALIDATION:
				collectionPostConversionValidators.add(annotationUsageResult);
				break;
			case CONVERT:
				converter = annotationUsageResult;
				break;
			case FORM_FIELD:
				formField = true;
				break;
			case MANUAL_PARAMETER_CONVERSION:
				manualParameterConversion = true;
				break;
			case NA:
				// Ignored
				break;
			case NON_CONVERT_VALIDATION:
				validators.add(annotationUsageResult);
				break;
            case POST_ADJUSTER:
                postConversionAdjusters.add(annotationUsageResult);
                break;
			case POST_VALIDATION:
				postConversionValidators.add(annotationUsageResult);
				break;
			}
		}

		public List<ConfiguredPolicy<T>> getAdjusters() {
			return adjusters;
		}
		public ConfiguredPolicy<T> getCollectionConverter() {
			return collectionConverter;
		}
        public List<ConfiguredPolicy<T>> getCollectionPostConversionAdjusters() {
            return collectionPostConversionAdjusters;
        }
		public List<ConfiguredPolicy<T>> getCollectionPostConversionValidators() {
			return collectionPostConversionValidators;
		}
		public ConfiguredPolicy<T> getConverter() {
			return converter;
		}
		public boolean getHasNoAnnotations() {
			return (adjusters.size() == 0) && (collectionConverter == null) && (collectionPostConversionValidators.size() == 0) &&
				(converter == null) && !formField && !manualParameterConversion && (postConversionValidators.size() == 0) && 
				(validators.size() == 0);
		}
		public boolean getFormField() {
			return formField;
		}
        public boolean getManualParameterConversion() {
			return manualParameterConversion;
		}
		public List<ConfiguredPolicy<T>> getPostConversionAdjusters() {
            return postConversionAdjusters;
        }
		public List<ConfiguredPolicy<T>> getPostConversionValidators() {
			return postConversionValidators;
		}
		public List<ConfiguredPolicy<T>> getValidators() {
			return validators;
		}


		@Override
		public String toString() {
			final int maxLen = 10;
			return "AnnotationEntries [adjusters="
					+ (adjusters != null ? adjusters.subList(0, Math.min(adjusters.size(), maxLen)) : null)
					+ ", collectionConverter=" + collectionConverter + ", collectionPostConversionAdjusters="
					+ (collectionPostConversionAdjusters != null ? collectionPostConversionAdjusters.subList(0,
							Math.min(collectionPostConversionAdjusters.size(), maxLen)) : null)
					+ ", collectionPostConversionValidators="
					+ (collectionPostConversionValidators != null
							? collectionPostConversionValidators.subList(0,
									Math.min(collectionPostConversionValidators.size(), maxLen))
							: null)
					+ ", converter=" + converter + ", formField=" + formField + ", manualParameterConversion="
					+ manualParameterConversion + ", postConversionAdjusters="
					+ (postConversionAdjusters != null
							? postConversionAdjusters.subList(0, Math.min(postConversionAdjusters.size(), maxLen))
							: null)
					+ ", postConversionValidators="
					+ (postConversionValidators != null
							? postConversionValidators.subList(0, Math.min(postConversionValidators.size(), maxLen))
							: null)
					+ ", validators="
					+ (validators != null ? validators.subList(0, Math.min(validators.size(), maxLen)) : null) + "]";
		}
		
	}
    
    public enum AnnotationUsage {
    	    ADJUSTER(Adjuster.class, 0, null, AbstractAdjusterSupport.class, 0, null), 
    	    COLLECTION_CONVERT(CollectionConverter.class, 0, 1, AbstractCollectionConverterSupport.class, 0, 1), 
    	    COLLECTION_POST_ADJUSTER(CollectionPostConversionAdjuster.class, 0, 1, AbstractCollectionPostConversionAdjusterSupport.class, 0, 1), 
    	    COLLECTION_POST_VALIDATION(CollectionPostConversionValidator.class, 0, 1, CollectionPostConversionValidator.class, 0, 1), 
    	    CONVERT(Converter.class, 0, 1, AbstractConverterSupport.class, 0, 1), 
    	    FORM_FIELD(null, 0, null, null, 0, null), 
    	    MANUAL_PARAMETER_CONVERSION(null, 0, null, null, 0, null), 
    	    NA(null, 0, null, null, 0, null), 
    	    NON_CONVERT_VALIDATION(NonConversionValidator.class, 0, null, AbstractNonConversionValidatorSupport.class, 0, null), 
    	    POST_ADJUSTER(PostConversionAdjuster.class, 0, 1, AbstractPostConversionAdjusterSupport.class, 0, 1), 
    	    POST_VALIDATION(PostConversionValidator.class, 0, 1, AbstractPostConversionValidatorSupport.class, 0, 1);
    	
        private Class<?> directInterface;
        private int directInterfaceAnnotationIndex;
        private Integer directInterfaceRecipientTypeIndex;
        private Class<?> supportClass;
        private int supportClassAnnotationIndex;
        private Integer supportClassRecipientTypeIndex;
        
        private AnnotationUsage(Class<?> directInterface, int directInterfaceAnnotationIndex, Integer directInterfaceRecipientTypeIndex, 
        		Class<?> supportClass, int supportClassAnnotationIndex, Integer supportClassRecipientTypeIndex) {
            this.directInterface = directInterface;
            this.directInterfaceAnnotationIndex = directInterfaceAnnotationIndex;
            this.directInterfaceRecipientTypeIndex = directInterfaceRecipientTypeIndex;
            this.supportClass = supportClass;
            this.supportClassAnnotationIndex = supportClassAnnotationIndex;
            this.supportClassRecipientTypeIndex = supportClassRecipientTypeIndex;
        }

        /**
         * Returns interface, such as {@link Converter}, that defines policies of this annotation usage, or null if not 
         * applicable.
         */
		public Class<?> getDirectInterface() {
			return directInterface;
		}

		/**
		 * Returns index of generic parameters of interface for annotation type, with 0 being the first. 
		 */
		public int getDirectInterfaceAnnotationIndex() {
			return directInterfaceAnnotationIndex;
		}

		/**
		 * Returns index of generic parameters of interface for recipient type, with 0 being the first, or null if not 
		 * applicable. 
		 */
		public Integer getDirectInterfaceRecipientTypeIndex() {
			return directInterfaceRecipientTypeIndex;
		}

        /**
         * Returns template class, such as {@link AbstractConverterSupport}, that aids policy implementation of this 
         * annotation usage, or null if not applicable.
         */
		public Class<?> getSupportClass() {
			return supportClass;
		}

		/**
		 * Returns index of generic parameters of support class for annotation type, with 0 being the first. 
		 */
		public int getSupportClassAnnotationIndex() {
			return supportClassAnnotationIndex;
		}

		/**
		 * Returns index of generic parameters of support class for recipient type, with 0 being the first, or null if 
		 * not applicable. 
		 */
		public Integer getSupportClassRecipientTypeIndex() {
			return supportClassRecipientTypeIndex;
		}

    }
    
	public static class CategoriseFieldResult {
    	private Collection<FieldUsage<?>> autoParameterConversionFields;	// Converted from parameter using annotated converter
    	private Collection<FieldUsage<?>> defaultParameterConversionFields;	// Converted from parameter using default converter
    	private Collection<FieldUsage<?>> fileFields;                       // Special parameter type created by FileUploadInterceptor
    	private Collection<FieldUsage<?>> ignoreFields;  					// Not annotated and ignore non-annotated fields option was in effect
    	private Collection<FieldUsage<?>> manualParameterConversionFields;  // Calls form to convert from parameter
    	private Collection<FieldUsage<?>> noConversionFields;               // Set from parameter and no conversion but other annotations can apply 
    	private Collection<PairFieldUsage<?>> pairConversionFields;         // Sets string field from parameter, than converts that to non-string field
    	private Collection<FieldUsage<?>> setOnlyFields;                    // Set from parameter and annotations are ignored
    	
    	
		public CategoriseFieldResult() {
	    	autoParameterConversionFields = new ArrayList<>();
	    	defaultParameterConversionFields = new ArrayList<>();
	    	fileFields = new ArrayList<>();
	    	ignoreFields = new ArrayList<>();
	    	manualParameterConversionFields = new ArrayList<>();
	    	noConversionFields = new ArrayList<>(); 
	    	pairConversionFields = new ArrayList<>();
	    	setOnlyFields = new ArrayList<>();
		}
		
		
		public Collection<FieldUsage<?>> getAutoParameterConversionFields() {
			return autoParameterConversionFields;
		}
		public void setAutoParameterConversionFields(Collection<FieldUsage<?>> autoParameterConversionFields) {
			this.autoParameterConversionFields = autoParameterConversionFields;
		}

		public Collection<FieldUsage<?>> getDefaultParameterConversionFields() {
			return defaultParameterConversionFields;
		}
		public void setDefaultParameterConversionFields(Collection<FieldUsage<?>> defaultParameterConversionFields) {
			this.defaultParameterConversionFields = defaultParameterConversionFields;
		}
		
		public Collection<FieldUsage<?>> getFileFields() {
			return fileFields;
		}
		public void setFileFields(Collection<FieldUsage<?>> fileFields) {
			this.fileFields = fileFields;
		}

		public Collection<FieldUsage<?>> getIgnoreFields() {
			return ignoreFields;
		}
		public void setIgnoreFields(Collection<FieldUsage<?>> ignoreFields) {
			this.ignoreFields = ignoreFields;
		}

		public Collection<FieldUsage<?>> getManualParameterConversionFields() {
			return manualParameterConversionFields;
		}
		public void setManualParameterConversionFields(Collection<FieldUsage<?>> manualParameterConversionFields) {
			this.manualParameterConversionFields = manualParameterConversionFields;
		}
		
		public Collection<FieldUsage<?>> getNoConversionFields() {
			return noConversionFields;
		}
		public void setNoConversionFields(Collection<FieldUsage<?>> noConversionFields) {
			this.noConversionFields = noConversionFields;
		}

		public Collection<PairFieldUsage<?>> getPairConversionFields() {
			return pairConversionFields;
		}
		public void setPairConversionFields(Collection<PairFieldUsage<?>> pairConversionFields) {
			this.pairConversionFields = pairConversionFields;
		}
		
		public Collection<FieldUsage<?>> getSetOnlyFields() {
			return setOnlyFields;
		}
		public void setSetOnlyFields(Collection<FieldUsage<?>> setOnlyFields) {
			this.setOnlyFields = setOnlyFields;
		}


		@Override
		public String toString() {
			final int maxLen = 10;
			return "CategoriseFieldResult [autoParameterConversionFields="
					+ (autoParameterConversionFields != null ? toString(autoParameterConversionFields, maxLen) : null)
					+ ", defaultParameterConversionFields="
					+ (defaultParameterConversionFields != null ? toString(defaultParameterConversionFields, maxLen)
							: null)
					+ ", fileFields=" + (fileFields != null ? toString(fileFields, maxLen) : null) + ", ignoreFields="
					+ (ignoreFields != null ? toString(ignoreFields, maxLen) : null)
					+ ", manualParameterConversionFields="
					+ (manualParameterConversionFields != null ? toString(manualParameterConversionFields, maxLen)
							: null)
					+ ", noConversionFields="
					+ (noConversionFields != null ? toString(noConversionFields, maxLen) : null)
					+ ", pairConversionFields="
					+ (pairConversionFields != null ? toString(pairConversionFields, maxLen) : null)
					+ ", setOnlyFields=" + (setOnlyFields != null ? toString(setOnlyFields, maxLen) : null) + "]";
		}


		private String toString(Collection<?> collection, int maxLen) {
			StringBuilder builder = new StringBuilder();
			builder.append("[");
			int i = 0;
			for (Iterator<?> iterator = collection.iterator(); iterator.hasNext() && i < maxLen; i++) {
				if (i > 0) {
					builder.append(", ");
				}
				builder.append(iterator.next());
			}
			builder.append("]");
			return builder.toString();
		}

    }
	
    /**
     * Describes a type of form field annotation and an instance of the policy it configures. 
     */
    public static class ConfiguredPolicy<T> implements Serializable {
        private static final long serialVersionUID = -3300262924248884585L;
        
        public static <T> ConfiguredPolicy<T> makeAdjusterResult(Adjuster<Annotation> adjuster) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.ADJUSTER;
        	result.annotation = adjuster.getAnnotation();
            result.adjuster = adjuster;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeCollectionConverterResult(CollectionConverter<?,T> collectionConverter, boolean defaultAnnotation) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.COLLECTION_CONVERT;
        	result.annotation = collectionConverter.getAnnotation();
        	result.defaultAnnotation = defaultAnnotation;
            result.collectionConverter = collectionConverter;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeCollectionPostConversionAdjusterResult(CollectionPostConversionAdjuster<?,T> collectionPostConversionAdjuster) {
            ConfiguredPolicy<T> result;
            
            result = new ConfiguredPolicy<T>();
            result.annotationUsage = AnnotationUsage.COLLECTION_POST_ADJUSTER;
            result.annotation = collectionPostConversionAdjuster.getAnnotation();
            result.collectionPostConversionAdjuster = collectionPostConversionAdjuster;
            return result;
        }
        public static <T> ConfiguredPolicy<T> makeCollectionPostConversionValidatorResult(CollectionPostConversionValidator<?,T> collectionPostConversionValidator) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.COLLECTION_POST_VALIDATION;
        	result.annotation = collectionPostConversionValidator.getAnnotation();
            result.collectionPostConversionValidator = collectionPostConversionValidator;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeConverterResult(Converter<?,T> converter, boolean defaultAnnotation) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.CONVERT;
        	result.annotation = converter.getAnnotation();
        	result.defaultAnnotation = defaultAnnotation;
            result.converter = converter;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeFormFieldResult(Annotation annotation) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.FORM_FIELD;
        	result.annotation = annotation;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeManualParameterConversionResult(Annotation annotation) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.MANUAL_PARAMETER_CONVERSION;
        	result.annotation = annotation;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeNAResult(Annotation annotation) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.NA;
        	result.annotation = annotation;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makeNonConversionValidatorResult(NonConversionValidator<?> nonConversionValidator) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.NON_CONVERT_VALIDATION;
        	result.annotation = nonConversionValidator.getAnnotation();
            result.nonConversionValidator = nonConversionValidator;
        	return result;
        }
        public static <T> ConfiguredPolicy<T> makePostConversionAdjusterResult(PostConversionAdjuster<?,T> postConversionAdjuster) {
            ConfiguredPolicy<T> result;
            
            result = new ConfiguredPolicy<T>();
            result.annotationUsage = AnnotationUsage.POST_ADJUSTER;
            result.annotation = postConversionAdjuster.getAnnotation();
            result.postConversionAdjuster = postConversionAdjuster;
            return result;
        }
        public static <T> ConfiguredPolicy<T> makePostConversionValidatorResult(PostConversionValidator<?,T> postConversionValidator) {
        	ConfiguredPolicy<T> result;
        	
        	result = new ConfiguredPolicy<T>();
        	result.annotationUsage = AnnotationUsage.POST_VALIDATION;
        	result.annotation = postConversionValidator.getAnnotation();
            result.postConversionValidator = postConversionValidator;
        	return result;
        }
        
        private AnnotationUsage annotationUsage;
        private Annotation annotation;
        private boolean defaultAnnotation;
        private Adjuster<?> adjuster;
        private CollectionConverter<?,T> collectionConverter;
        private CollectionPostConversionAdjuster<?,T> collectionPostConversionAdjuster;
        private CollectionPostConversionValidator<?,T> collectionPostConversionValidator;
        private Converter<?,T> converter;
        private NonConversionValidator<?> nonConversionValidator;
        private PostConversionAdjuster<?,T> postConversionAdjuster;
        private PostConversionValidator<?,T> postConversionValidator;
        
        private ConfiguredPolicy() {
            super();
            this.annotationUsage = AnnotationUsage.NA;
        	this.annotation = null;
            this.adjuster = null;
            this.collectionConverter = null;
            this.collectionPostConversionAdjuster = null;
            this.collectionPostConversionValidator = null;
            this.converter = null;
            this.nonConversionValidator = null;
            this.postConversionAdjuster = null;
            this.postConversionValidator = null;
        }
        
        
        public Adjuster<?> getAdjuster() {
            return adjuster;
        }
        public Annotation getAnnotation() {
            return annotation;
        }
		public AnnotationUsage getAnnotationUsage() {
            return annotationUsage;
        }
        public CollectionConverter<?,T> getCollectionConverter() {
            return collectionConverter;
        }
        public CollectionPostConversionAdjuster<?, T> getCollectionPostConversionAdjuster() {
            return collectionPostConversionAdjuster;
        }
        public CollectionPostConversionValidator<?,T> getCollectionPostConversionValidator() {
            return collectionPostConversionValidator;
        }
        public Converter<?,T> getConverter() {
            return converter;
        }
        public boolean getDefaultAnnotation() {
			return defaultAnnotation;
		}
        public NonConversionValidator<?> getNonConversionValidator() {
            return nonConversionValidator;
        }
        public PostConversionAdjuster<?, T> getPostConversionAdjuster() {
            return postConversionAdjuster;
        }
        public PostConversionValidator<?,T> getPostConversionValidator() {
            return postConversionValidator;
        }
		@Override
		public String toString() {
			return "ConfiguredPolicy [" + (annotationUsage != null ? "annotationUsage=" + annotationUsage + ", " : "")
					+ (annotation != null ? "annotation=" + annotation + ", " : "") + "defaultAnnotation="
					+ defaultAnnotation + ", " + (adjuster != null ? "adjuster=" + adjuster + ", " : "")
					+ (collectionConverter != null ? "collectionConverter=" + collectionConverter + ", " : "")
					+ (collectionPostConversionAdjuster != null
							? "collectionPostConversionAdjuster=" + collectionPostConversionAdjuster + ", "
							: "")
					+ (collectionPostConversionValidator != null
							? "collectionPostConversionValidator=" + collectionPostConversionValidator + ", "
							: "")
					+ (converter != null ? "converter=" + converter + ", " : "")
					+ (nonConversionValidator != null ? "nonConversionValidator=" + nonConversionValidator + ", " : "")
					+ (postConversionAdjuster != null ? "postConversionAdjuster=" + postConversionAdjuster + ", " : "")
					+ (postConversionValidator != null ? "postConversionValidator=" + postConversionValidator : "")
					+ "]";
		}

    }
    
    /**
     * Describes table entry that finds a default converter for a single value field. 
     */
    public static class DefaultCollectionConverterEntry<T,C extends CollectionConverter<?,T>> {
        private Class<T> itemClass;
        private Class<C> collectionConverterClass;
        private boolean builtIn;
        
        
        public DefaultCollectionConverterEntry(Class<T> itemClass, Class<C> collectionConverterClass, boolean builtIn) {
            super();
            this.itemClass = itemClass;
            this.collectionConverterClass = collectionConverterClass;
            this.builtIn = builtIn;
        }
        
        
		public boolean getBuiltIn() {
			return builtIn;
		}
		public Class<C> getCollectionConverterClass() {
			return collectionConverterClass;
		}
		public Class<T> getItemClass() {
			return itemClass;
		}


		@Override
		public String toString() {
			return "DefaultCollectionConverterEntry [" + (itemClass != null ? "itemClass=" + itemClass + ", " : "")
					+ (collectionConverterClass != null ? "collectionConverterClass=" + collectionConverterClass + ", "
							: "")
					+ "builtIn=" + builtIn + "]";
		}


		@Override
		public int hashCode() {
			return Objects.hash(builtIn, collectionConverterClass, itemClass);
		}


		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			DefaultCollectionConverterEntry<?,?> other = (DefaultCollectionConverterEntry<?,?>) obj;
			return builtIn == other.builtIn && Objects.equals(collectionConverterClass, other.collectionConverterClass)
					&& Objects.equals(itemClass, other.itemClass);
		}
        
    }
    
    /**
     * Describes table entry that finds a default converter for a single value field. 
     */
    public static class DefaultConverterEntry<T,C extends Converter<?,T>> {
        private Class<T> fieldClass;
        private Class<C> converterClass;
        private boolean builtIn;
        
        
        public DefaultConverterEntry(Class<T> fieldClass, Class<C> converterClass, boolean builtIn) {
            super();
            this.fieldClass = fieldClass;
            this.converterClass = converterClass;
            this.builtIn = builtIn;
        }
        
        
        public boolean getBuiltIn() {
			return builtIn;
		}
		public Class<C> getConverterClass() {
			return converterClass;
		}
		public Class<T> getFieldClass() {
			return fieldClass;
		}


		@Override
		public String toString() {
			return "DefaultConverterEntry [" + (fieldClass != null ? "fieldClass=" + fieldClass + ", " : "")
					+ (converterClass != null ? "converterClass=" + converterClass + ", " : "") + "builtIn=" + builtIn
					+ "]";
		}


		@Override
		public int hashCode() {
			return Objects.hash(builtIn, converterClass, fieldClass);
		}


		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			DefaultConverterEntry<?,?> other = (DefaultConverterEntry<?,?>) obj;
			return builtIn == other.builtIn && Objects.equals(converterClass, other.converterClass)
					&& Objects.equals(fieldClass, other.fieldClass);
		}
        
    }
    
    public static class FieldUsage<T> {
    	private AnnotationEntries<T> annountationEntries;
    	private Field field;
    	private String name;
    	private Class<T> fieldType;
    	private boolean collection;
    	private boolean array;
    	
    	
    	public FieldUsage() {
    		super();
    	}
    	
    	
    	public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		/**
		 * Returns true if form field is an array. 
		 */
		public boolean getArray() {
			return array;
		}

		/**
		 * Returns true if form field is a collection. 
		 */
		public boolean getCollection() {
			return collection;
		}
		public Field getField() {
			return field;
		}
		
		/**
		 * Returns form field's class for single value type, component type for arrays, or generic type for collections. 
		 */
		public Class<T> getFieldType() {
			return fieldType;
		}
		public String getName() {
			return name;
		}

		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		public void setArray(boolean array) {
			this.array = array;
		}

		public void setCollection(boolean collection) {
			this.collection = collection;
		}
		public void setField(Field field) {
			this.field = field;
		}

		public void setFieldType(Class<T> fieldType) {
			this.fieldType = fieldType;
		}
		public void setName(String name) {
			this.name = name;
		}


		@Override
		public String toString() {
			return "FieldUsage [annountationEntries=" + annountationEntries + ", field=" + field + ", name=" + name
					+ ", fieldType=" + fieldType + ", collection=" + collection + ", array=" + array + "]";
		}
    	
    }
    
    public enum FormFieldCategory {FILE, MANUAL_CONVERSION, NON_STRING, NON_STRING_ARRAY, NON_STRING_COLLECTION, STRING, STRING_ARRAY, STRING_COLLECTION}
    
    public static class PairFieldUsage<T> {
    	private AnnotationEntries<T> annountationEntries;
    	private Field formattedField;
    	private String formattedName;
    	private Field unformattedField;
    	private String unformattedName;
    	
    	
		public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		public Field getFormattedField() {
			return formattedField;
		}
		
		public String getFormattedName() {
			return formattedName;
		}
		public Field getUnformattedField() {
			return unformattedField;
		}
		
		public String getUnformattedName() {
			return unformattedName;
		}
		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		
		public void setFormattedField(Field formattedField) {
			this.formattedField = formattedField;
		}
		public void setFormattedName(String formattedName) {
			this.formattedName = formattedName;
		}
		
		public void setUnformattedField(Field unformattedField) {
			this.unformattedField = unformattedField;
		}
		public void setUnformattedName(String unformattedName) {
			this.unformattedName = unformattedName;
		}
		
		
		@Override
		public String toString() {
			return "PairFieldUsage [annountationEntries=" + annountationEntries + ", formattedField=" + formattedField
					+ ", formattedName=" + formattedName + ", unformattedField=" + unformattedField
					+ ", unformattedName=" + unformattedName + "]";
		}
		
    }
    
    /**
     * Describes table entry that finds a policy class from the annotation class that configures it. 
     */
    public static class PolicyEntry<A extends Annotation,P extends Policy<A>,T> {
        private Class<A> annotationClass;
        private Class<P> policyClass;
        private Class<T> recipientType;
        private AnnotationUsage annotationUsage;
        private boolean builtIn;
        private boolean defaultPolicy;
        
        
        public PolicyEntry(Class<A> annotationClass, Class<P> policyClass, Class<T> recipientType, AnnotationUsage annotationUsage,
        		boolean builtIn, boolean defaultPolicy) {
            super();
            this.annotationClass = annotationClass;
            this.policyClass = policyClass;
            this.recipientType = recipientType;
            this.annotationUsage = annotationUsage;
            this.builtIn = builtIn;
            this.defaultPolicy = defaultPolicy;
        }
        
        
        public Class<A> getAnnotationClass() {
            return annotationClass;
        }
		public Class<P> getPolicyClass() {
            return policyClass;
        }
		public Class<T> getRecipientType() {
			return recipientType;
		}
        public AnnotationUsage getAnnotationUsage() {
			return annotationUsage;
		}
		public boolean getBuiltIn() {
			return builtIn;
		}
		public boolean getDefaultPolicy() {
			return defaultPolicy;
		}


		@Override
		public int hashCode() {
			return Objects.hash(annotationClass, annotationUsage, builtIn, defaultPolicy, policyClass, recipientType);
		}


		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			PolicyEntry<?,?,?> other = (PolicyEntry<?,?,?>) obj;
			return Objects.equals(annotationClass, other.annotationClass) && annotationUsage == other.annotationUsage
					&& builtIn == other.builtIn && defaultPolicy == other.defaultPolicy
					&& Objects.equals(policyClass, other.policyClass)
					&& Objects.equals(recipientType, other.recipientType);
		}


		@Override
		public String toString() {
			return "PolicyEntry [" + (annotationClass != null ? "annotationClass=" + annotationClass + ", " : "")
					+ (policyClass != null ? "policyClass=" + policyClass + ", " : "")
					+ (recipientType != null ? "recipientType=" + recipientType + ", " : "")
					+ (annotationUsage != null ? "annotationUsage=" + annotationUsage + ", " : "") + "builtIn="
					+ builtIn + ", defaultPolicy=" + defaultPolicy + "]";
		}
		
    }
    
    private static final Logger LOG = LogManager.getLogger(InterceptorCommonLibrary.class);
    
    private static final String ACTION_CONTEXT_FORMATTED_FORMS = InterceptorCommonLibrary.class.getName() + ".formattedForms";
    
    private static Set<Class<? extends Annotation>> customAnnotations = Collections.unmodifiableSet(new HashSet<>(
    		Arrays.asList(CustomAdjuster.class, CustomCollectionConversion.class, CustomCollectionPostConversionAdjuster.class,
    				CustomCollectionPostConversionValidation.class, CustomConversion.class, CustomPostConversionAdjuster.class,
    				CustomPostConversionValidation.class, CustomValidation.class)));
    
    public static final String DEFAULT_ACCEPT_CLASSES 						= "";
    public static final String DEFAULT_ACCEPT_PACKAGES 						= "";
    public static final String DEFAULT_CLASSPATH_SCANNING_REPLACE_BUILT_IN	= "false";
    public static final String DEFAULT_ENABLE_CLASSPATH_SCANNING 			= "false";
    public static final String DEFAULT_REJECT_CLASSES 						= "";
    public static final String DEFAULT_REJECT_PACKAGES 						= "";
    
    public static final String STRUTS_CONSTANT_ACCEPT_CLASSES 						= "name.matthewgreet.strutscommons.accept_classes";
    public static final String STRUTS_CONSTANT_ACCEPT_PACKAGES 						= "name.matthewgreet.strutscommons.accept_packages";
    public static final String STRUTS_CONSTANT_CLASSPATH_SCANNING_REPLACE_BUILT_IN	= "name.matthewgreet.strutscommons.classpath_scanning_replace_built_in";
    public static final String STRUTS_CONSTANT_ENABLE_CLASSPATH_SCANNING 			= "name.matthewgreet.strutscommons.enable_classpath_scanning";
    public static final String STRUTS_CONSTANT_REJECT_CLASSES 						= "name.matthewgreet.strutscommons.reject_classes";
    public static final String STRUTS_CONSTANT_REJECT_PACKAGES 						= "name.matthewgreet.strutscommons.reject_packages";
    
    /**
     * Returns formatted text from unformatted value using validator, or empty string if unformatted value is null.
     */
    private static <T> String formatCollectionValue(CollectionConverter<?,T> collectionConverter, Collection<T> unformattedValue) throws Exception {
        if (unformattedValue != null) {
            return collectionConverter.format(unformattedValue);
        } else {
            return "";
        }
    }
    
    /**
     * Returns formatted text from unformatted value using validator, or empty string if unformatted value is null.
     */
    private static <T> String formatValue(Converter<?,T> converter, T unformattedValue) throws Exception {
        if (unformattedValue != null) {
            return converter.format(unformattedValue);
        } else {
            return "";
        }
    }
    
    /**
     * Finds field that has unformatted value from fields found in the form, or null if not found. 
     */
    private static Field getUnformattedField(Collection<Field> fields, String unformattedFieldName) {
        Field result;
        
        result = null;
        for (Field field: fields) {
            if (field.getName().equals(unformattedFieldName)) {
                result = field;
                break;
            }
        }
        return result;
    }
    
    /**
     * Sets how each form field should be converted.
     */
    @SuppressWarnings("unchecked")
	public static <T> CategoriseFieldResult categoriseFormFields(Collection<Field> formFields, PolicyLookup policyLookup, boolean ignoreNonAnnotatedField) {
    	AnnotationEntries<T> annotationEntries;
    	CategoriseFieldResult result;
    	CollectionConverter<?,T> defaultCollectionConverter;
    	Converter<?,T> defaultConverter;
    	FormFieldCategory fieldCategory;
    	FieldUsage<T> fieldUsage, unformattedFieldUsage;
    	PairFieldUsage<T> pairFieldUsage;
    	Map<String,FieldUsage<T>> allFieldUsages, autoConversionFieldUsages, defaultConversionFieldUsages, fileFieldUsages;
    	Map<String,FieldUsage<T>> ignoreFieldUsages, manualConversionFieldUsages, noConversionFieldUsages;
    	Map<String,FieldUsage<T>> pairConversionFieldUsages, setOnlyFieldUsages;
    	Class<?> unformattedFieldClass;
    	String unformattedFieldName;
    	boolean explicitConversion, manualConversion, failed;

    	result = new CategoriseFieldResult();
    	allFieldUsages = new HashMap<>();
    	for (Field formField: formFields) {
    		try {
				fieldUsage = getFormFieldUsage(formField, policyLookup);
	    		allFieldUsages.put(fieldUsage.getName(), fieldUsage);
			}
    		catch (FormFieldAnnotationTypeMismatchException e) {
                LOG.error("Form field annotation is incompatible with form field type and will be skipped" + 
    		        "  form=" + formField.getDeclaringClass() + "  field=" + formField.getName() + 
    		        "  annotation=" + e.getAnnotation().annotationType());
			}
    	}
    	autoConversionFieldUsages = new HashMap<>();
    	defaultConversionFieldUsages = new HashMap<>();
    	fileFieldUsages = new HashMap<>();
    	ignoreFieldUsages = new HashMap<>();
    	noConversionFieldUsages = new HashMap<>();
    	manualConversionFieldUsages = new HashMap<>();
    	pairConversionFieldUsages = new HashMap<>();
    	setOnlyFieldUsages = new HashMap<>();
    	
    	for (FieldUsage<T> fieldUsage2: allFieldUsages.values()) {
    		if (!ignoreNonAnnotatedField || !fieldUsage2.getAnnountationEntries().getHasNoAnnotations()) {
	    		explicitConversion = fieldUsage2.getAnnountationEntries().getConverter() != null || 
	    		    fieldUsage2.getAnnountationEntries().getCollectionConverter() != null;
	        	manualConversion = fieldUsage2.getAnnountationEntries().getManualParameterConversion();
	        	fieldCategory = getFormFieldCategory(fieldUsage2);
	    		
	        	switch (fieldCategory) {
	        	case FILE:
	        		fileFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
	        		break;
	            case MANUAL_CONVERSION:
	                manualConversion = true;
	                manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
	                break;
				case NON_STRING:
					if (manualConversion) {
		                manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else if (explicitConversion) {
		    			autoConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else {
		    			defaultConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					}
					break;
				case NON_STRING_ARRAY:
	                manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					break;
				case NON_STRING_COLLECTION:
					if (manualConversion) {
		                manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else if (explicitConversion) {
		    			autoConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
				    } else {
		                defaultConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
				    }
					break;
				case STRING:
					if (manualConversion) {
		                manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else if (explicitConversion) {
		    			pairConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else {
		    			noConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					}
					break;
				case STRING_ARRAY:
					setOnlyFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					break;
				case STRING_COLLECTION:
					if (manualConversion) {
		                setOnlyFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else if (explicitConversion) {
		    			autoConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					} else {
		                setOnlyFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
					}
					break;
	        	}
    		} else {
    			ignoreFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
    		}
    	}

    	// Match formatted/unformatted pairs
    	for (FieldUsage<T> formattedFieldUsage: pairConversionFieldUsages.values()) {
    		if (formattedFieldUsage.getAnnountationEntries().getCollectionConverter() != null) {
    			unformattedFieldName = formattedFieldUsage.getAnnountationEntries().getCollectionConverter().getCollectionConverter().getRecipientFieldName();
    		} else {
    			unformattedFieldName = formattedFieldUsage.getAnnountationEntries().getConverter().getConverter().getRecipientFieldName();
    		}
            if (unformattedFieldName == null || unformattedFieldName.length() == 0) {
            	unformattedFieldName = getDefaultRecipientName(formattedFieldUsage.getName());
            }
            failed = false;
            unformattedFieldUsage = defaultConversionFieldUsages.get(unformattedFieldName);
            if (unformattedFieldUsage == null) {
                failed = true;
                noConversionFieldUsages.put(formattedFieldUsage.getName(), formattedFieldUsage);
                LOG.error("Form field annotation refers to non-existent or annotated unformatted field, explicitly or by default, and will be treated as no conversion" + 
       		        "  form=" + formattedFieldUsage.getField().getDeclaringClass() + 
                    "  field=" + formattedFieldUsage.getName() + "  unformattedFieldName=" + unformattedFieldName);
            }
            if (!failed) {
    			unformattedFieldClass = formattedFieldUsage.getField().getType();
    			annotationEntries = unformattedFieldUsage.getAnnountationEntries();
        		if (annotationEntries.getConverter() != null && !annotationEntries.getConverter().getConverter().getRecipientClass().isAssignableFrom(unformattedFieldClass)) {
                    failed = true;
                    manualConversionFieldUsages.put(formattedFieldUsage.getName(), formattedFieldUsage);
                    manualConversionFieldUsages.put(unformattedFieldUsage.getName(), unformattedFieldUsage);
        			defaultConversionFieldUsages.remove(unformattedFieldName);
                    LOG.error("Unformatted field of formatted/unformatted pair is incompatible with converter, and will be treated as manual parameter conversion" + 
           		        "  form=" + formattedFieldUsage.getField().getDeclaringClass() + 
                        "  formatted field=" + formattedFieldUsage.getName() + "  unformatted field=" + unformattedFieldName);
        		} else if (annotationEntries.getCollectionConverter() != null && 
        				!annotationEntries.getCollectionConverter().getCollectionConverter().getRecipientClass().isAssignableFrom(unformattedFieldClass)) {
                    failed = true;
                    manualConversionFieldUsages.put(formattedFieldUsage.getName(), formattedFieldUsage);
                    manualConversionFieldUsages.put(unformattedFieldUsage.getName(), unformattedFieldUsage);
        			defaultConversionFieldUsages.remove(unformattedFieldName);
                    LOG.error("Unformatted field of formatted/unformatted pair is incompatible with converter, and will be treated as manual parameter conversion" + 
           		        "  form=" + formattedFieldUsage.getField().getDeclaringClass() + 
                        "  formatted field=" + formattedFieldUsage.getName() + "  unformatted field=" + unformattedFieldName);
        		}
        		for (ConfiguredPolicy<T> configuredPolicy: annotationEntries.getPostConversionValidators()) {
            		if (!configuredPolicy.getPostConversionValidator().getRecipientClass().isAssignableFrom(unformattedFieldClass)) {
                        failed = true;
                        manualConversionFieldUsages.put(formattedFieldUsage.getName(), formattedFieldUsage);
                        manualConversionFieldUsages.put(unformattedFieldUsage.getName(), unformattedFieldUsage);
            			defaultConversionFieldUsages.remove(unformattedFieldName);
                        LOG.error("Unformatted field of formatted/unformatted pair is incompatible with post-conversion validator, and will be treated as manual parameter conversion" + 
               		        "  form=" + formattedFieldUsage.getField().getDeclaringClass() + 
                            "  formatted field=" + formattedFieldUsage.getName() + "  unformatted field=" + unformattedFieldName);
                        break;
            		}
        		}
        		for (ConfiguredPolicy<T> configuredPolicy: annotationEntries.getCollectionPostConversionValidators()) {
            		if (!configuredPolicy.getCollectionPostConversionValidator().getRecipientClass().isAssignableFrom(unformattedFieldClass)) {
                        failed = true;
                        manualConversionFieldUsages.put(formattedFieldUsage.getName(), formattedFieldUsage);
                        manualConversionFieldUsages.put(unformattedFieldUsage.getName(), unformattedFieldUsage);
            			defaultConversionFieldUsages.remove(unformattedFieldName);
                        LOG.error("Unformatted field of formatted/unformatted pair is incompatible with post-conversion validator, and will be treated as manual parameter conversion" + 
               		        "  form=" + formattedFieldUsage.getField().getDeclaringClass() + 
                            "  formatted field=" + formattedFieldUsage.getName() + "  unformatted field=" + unformattedFieldName);
                        break;
            		}
        		}
            }
            if (!failed) {
    			pairFieldUsage = new PairFieldUsage<T>();
    			pairFieldUsage.setAnnountationEntries(formattedFieldUsage.getAnnountationEntries());
    			pairFieldUsage.setUnformattedName(unformattedFieldUsage.getName());
    			pairFieldUsage.setUnformattedField(unformattedFieldUsage.getField());
    			pairFieldUsage.setFormattedName(formattedFieldUsage.getName());
    			pairFieldUsage.setFormattedField(formattedFieldUsage.getField());
    			result.getPairConversionFields().add(pairFieldUsage);
    			defaultConversionFieldUsages.remove(unformattedFieldName);
    		}
    	}
    	// Check non-string, no converter annotation have default converter
    	for (FieldUsage<T> fieldUsage2: new ArrayList<>(defaultConversionFieldUsages.values())) {
            if (Collection.class.isAssignableFrom(fieldUsage2.getField().getType())) {
            	defaultCollectionConverter = (CollectionConverter<?, T>)getDefaultCollectionConverter(fieldUsage2.getField(), policyLookup);
            	if (defaultCollectionConverter != null) {
            		fieldUsage2.getAnnountationEntries().addAnnotationUsageResult(ConfiguredPolicy.makeCollectionConverterResult(defaultCollectionConverter, true));
            		result.getDefaultParameterConversionFields().add(fieldUsage2);
            	} else {
            		manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
            		defaultConversionFieldUsages.remove(fieldUsage2.getName());
            	}
            } else {
            	defaultConverter = (Converter<?, T>)getDefaultConverter(fieldUsage2.getField(), policyLookup);
            	if (defaultConverter != null) {
            		fieldUsage2.getAnnountationEntries().addAnnotationUsageResult(ConfiguredPolicy.makeConverterResult(defaultConverter, true));
            		result.getDefaultParameterConversionFields().add(fieldUsage2);
            	} else {
            		manualConversionFieldUsages.put(fieldUsage2.getName(), fieldUsage2);
            		defaultConversionFieldUsages.remove(fieldUsage2.getName());
            	}
            }
    	}
    	// Add rest to result
    	result.getAutoParameterConversionFields().addAll(autoConversionFieldUsages.values());
    	result.getFileFields().addAll(fileFieldUsages.values());
    	result.getIgnoreFields().addAll(ignoreFieldUsages.values());
    	result.getManualParameterConversionFields().addAll(manualConversionFieldUsages.values());
    	result.getNoConversionFields().addAll(noConversionFieldUsages.values());
    	result.getSetOnlyFields().addAll(setOnlyFieldUsages.values());
    	return result;
	}

    /**
     * Returns whether converting validator can convert form field value to the recipient field's type. 
     */
    public static <T> boolean checkCollectionRecipientDataType(Field recipientField, Class<T> recipientClass) {
        Class<T> entryClass;

        entryClass = getTypeFromCollectionField(recipientField);
        if (entryClass != null) {
            return checkFieldClass(recipientClass, entryClass);
        } else {
        	return false;
        }
    }
    
    /**
     * Returns whether a converter's target data type can convert to the form field's data type. 
     */
    public static boolean checkFieldClass(Class<?> conversionClass, Class<?> fieldClass) {
        if (conversionClass == Boolean.class) {
            return Boolean.class.isAssignableFrom(fieldClass) || boolean.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Byte.class) {
            return Byte.class.isAssignableFrom(fieldClass) || byte.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Character.class) {
            return Character.class.isAssignableFrom(fieldClass) || char.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Double.class) {
            return Double.class.isAssignableFrom(fieldClass) || double.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Float.class) {
            return Float.class.isAssignableFrom(fieldClass) || float.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Integer.class) {
            return Integer.class.isAssignableFrom(fieldClass) || int.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Long.class) {
            return Long.class.isAssignableFrom(fieldClass) || long.class.isAssignableFrom(fieldClass);
        } else if (conversionClass == Short.class) {
            return Short.class.isAssignableFrom(fieldClass) || short.class.isAssignableFrom(fieldClass);
        } else {
            return conversionClass.isAssignableFrom(fieldClass);
        }
    }

    /**
     * Returns whether converting validator can convert form field value to the recipient field's type. 
     */
    public static boolean checkRecipientDataType(Field recipientField, Class<?> recipientClass) {
        Class<?> recipientFieldClass;
        
        recipientFieldClass = recipientField.getType();
        return checkFieldClass(recipientClass, recipientFieldClass);
    }
    
    /**
     * Returns whether a viewer Action's member field is definitely not a form. 
     */
    public static boolean fieldIsNotForm(Field actionField) {
		Class<?> fieldClass;

		// Simply checking for the Form annotation seems obvious but the annotation isn't mandatory
		fieldClass = actionField.getType();
		if (fieldClass.isArray()) {
			return true;
		}
		if (fieldClass.isPrimitive()) {
			return true;
		}
		if (Modifier.isStatic(actionField.getModifiers())) {
			return true;
		}
		if (String.class.isAssignableFrom(fieldClass)) {
			return true;
		}
		if (Collection.class.isAssignableFrom(fieldClass)) {
			return true;
		}
		return false;
	}

    /**
     * Returns whether field of an Action received a stored form. 
     */
    public static boolean fieldReceivedStoredForm(Field actionField) {
        StoredForm storedForm;
        
        if (actionField == null) {
            return false;
        }
        storedForm = getStoredForm();
        if (storedForm == null) {
            return false;
        }
        if (storedForm.getOwningURL() == null) {
            return false;
        }
        if (actionField.getDeclaringClass() != storedForm.getOwningActionClass()) {
            return false;
        }
        
        return storedForm.getOwningActionFieldNames().contains(actionField.getName());
    }

    /**
     * Returns form member fields that didn't have a conversion error.
     */
	public static Collection<Field> filterNonConversionErrorFormFields(Collection<Field> formFields, Map<String, ConversionData> conversionErrors) {
		Collection<Field> result;
		
        result = new ArrayList<>();
        for (Field field: formFields) {
        	if (conversionErrors == null || !conversionErrors.containsKey(field.getName())) {
        		result.add(field);
        	}
        }
        return result;
	}
	
	/**
	 * Returns form member fields that can plausibly receive form data.  This excludes static and final fields.
	 */
    public static Collection<Field> filterPlausibleFormFields(Collection<Field> formFields) {
		Collection<Field> result;
        int fieldModifiers;
		
        result = new ArrayList<>();
        for (Field field: formFields) {
        	fieldModifiers = field.getModifiers();
        	if (!Modifier.isFinal(fieldModifiers) && !Modifier.isStatic(fieldModifiers)) {
        		result.add(field);
        	}
        }
        return result;
	}
    
    /**
     * Sets text for not yet set form field. 
     */
    @SuppressWarnings("unchecked")
    public static <T> void formatCollectionFormField(Object form, Collection<Field> allFormFields, Field formField, ConfiguredPolicy<T> configuredPolicy) {
        Field unformattedField;
        String fieldName, unformattedFieldName, formattedText;
        Collection<T> unformattedValue;
        boolean compatible;
        
        fieldName = formField.getName();
        unformattedFieldName = configuredPolicy.getCollectionConverter().getRecipientFieldName();
        if (unformattedFieldName == null || unformattedFieldName.length() == 0) {
            unformattedFieldName = getDefaultRecipientName(fieldName);
        }
        unformattedField = getUnformattedField(allFormFields, unformattedFieldName);
        if (unformattedField == null) {
            LOG.error("Form field annotation refers to non-existent unformatted field, explicitly or by default" + 
                "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName);
            return;
        }
        compatible = checkCollectionRecipientDataType(unformattedField, configuredPolicy.getCollectionConverter().getRecipientClass());
        if (!compatible) {
            LOG.error("Unformatted field is not compatible with validator" + 
                    "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                    "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName + 
                    "  allowed class=" + configuredPolicy.getCollectionConverter().getRecipientClass());
            return;
        }

        unformattedValue = null;
        try {
            unformattedField.setAccessible(true);
            unformattedValue = (Collection<T>)unformattedField.get(form);
            formattedText = formatCollectionValue(configuredPolicy.getCollectionConverter(), unformattedValue);
            formField.setAccessible(true);
            formField.set(form, formattedText);
        }
        catch (Exception e) {
            LOG.error("Setting formatted form field failed" + 
                    "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                    "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName + 
                    "  value to format=" + unformattedValue, e);
        }
    }
    
    /**
     * Sets text for not yet set form field. 
     */
    @SuppressWarnings("unchecked")
    public static <T> void formatFormField(Object form, Collection<Field> allFormFields, Field formField, ConfiguredPolicy<T> configuredPolicy) {
        Field unformattedField;
        String fieldName, unformattedFieldName, formattedText;
        T unformattedValue;
        boolean compatible;
        
        fieldName = formField.getName();
        unformattedFieldName = configuredPolicy.getConverter().getRecipientFieldName();
        if (unformattedFieldName == null || unformattedFieldName.length() == 0) {
            unformattedFieldName = getDefaultRecipientName(fieldName);
        }
        unformattedField = getUnformattedField(allFormFields, unformattedFieldName);
        if (unformattedField == null) {
            LOG.error("Form field annotation refers to non-existent unformatted field, explicitly or by default" + 
                "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName);
            return;
        }
        compatible = checkRecipientDataType(unformattedField, configuredPolicy.getConverter().getRecipientClass());
        if (!compatible) {
            LOG.error("Unformatted field is not compatible with validator" + 
                    "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                    "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName + 
                    "  allowed class=" + configuredPolicy.getConverter().getRecipientClass());
            return;
        }

        unformattedValue = null;
        try {
            unformattedField.setAccessible(true);
            unformattedValue = (T)unformattedField.get(form);
            formattedText = formatValue(configuredPolicy.getConverter(), unformattedValue);
            formField.setAccessible(true);
            formField.set(form, formattedText);
        }
        catch (Exception e) {
            LOG.error("Setting formatted form field failed" + 
                    "  Struts action=" + ActionContext.getContext().getActionName() + "  form=" + form.getClass() + 
                    "  field=" + formField.getName() + "  unformattedFieldName=" + unformattedFieldName + 
                    "  value to format=" + unformattedValue, e);
        }
    }
    
    /**
     * Returns the formatted forms created by {@link FormFormatterInterceptor}.  As these are only created by a Struts 2 
     * pre-result listener set by the Interceptor, no result is available till then.  This is meant for the Interceptor
     * and Actions should use {@link StrutsMiscellaneousLibrary#getFormattedForms} as it's easier to use.  
     */
    @SuppressWarnings("unchecked")
	public static synchronized Map<String,Map<String,Object>> getActionContextFormattedForms() {
    	Map<String,Map<String,Object>> result;
    	
    	result = (Map<String, Map<String, Object>>)ActionContext.getContext().getContextMap().get(ACTION_CONTEXT_FORMATTED_FORMS);
    	if (result == null) {
    		result = new HashMap<>();
    		ActionContext.getContext().getContextMap().put(ACTION_CONTEXT_FORMATTED_FORMS, result);
    	}
    	return result;
    }

	/**
	 * Returns all recognised annotations and their linked policies.
	 */
	public static <T> AnnotationEntries<T> getAnnotationEntries(Field formField, PolicyLookup policyLookup) {
		AnnotationEntries<T> result;
		ConfiguredPolicy<T> configuredPolicy;
		
		result = new AnnotationEntries<T>();
        for (Annotation annotation: formField.getAnnotations()) {
            try {
            	configuredPolicy = getConfiguredPolicy(annotation, policyLookup);
            }
            catch (Exception e) {
            	configuredPolicy = ConfiguredPolicy.makeNAResult(annotation);
                LOG.error("Creation of linked policy for form field failed " + 
                    "  form (or helper) class=" + formField.getDeclaringClass() + "  field=" + formField.getName() + 
                    "  annotation=" + annotation.annotationType(), e);
            }
            
            result.addAnnotationUsageResult(configuredPolicy);
        }
        return result;
	}
	
	/**
	 * Returns category of form field policy implementation class or NA if not applicable.  
	 */
	public static AnnotationUsage getAnnotationUsageFromPolicyClass(Class<?> policyClass) {
		if (Adjuster.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.ADJUSTER;
		} else if (CollectionConverter.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.COLLECTION_CONVERT;
		} else if (CollectionPostConversionAdjuster.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.COLLECTION_POST_ADJUSTER;
		} else if (CollectionPostConversionValidator.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.COLLECTION_POST_VALIDATION;
		} else if (Converter.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.CONVERT;
		} else if (NonConversionValidator.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.NON_CONVERT_VALIDATION;
		} else if (PostConversionAdjuster.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.POST_ADJUSTER;
		} else if (PostConversionValidator.class.isAssignableFrom(policyClass)) {
			return AnnotationUsage.POST_VALIDATION;
		} else {
			return AnnotationUsage.NA;
		}
	}
	
    /**
     * Returns built-in single value converters. 
     */
	public static List<DefaultCollectionConverterEntry<?,?>> getBuiltInDefaultCollectionConverters() {
    	List<DefaultCollectionConverterEntry<?,?>> result;
    	
    	result = new ArrayList<>();
    	result.add(new DefaultCollectionConverterEntry<>(Integer.class, IntegerCSVConverter.class, true));
    	result.add(new DefaultCollectionConverterEntry<>(String.class, StringCSVConverter.class, true));
    	return result;
	}
	
    /**
     * Returns built-in single value converters. 
     */
	public static List<DefaultConverterEntry<?,?>> getBuiltInDefaultConverters() {
    	List<DefaultConverterEntry<?,?>> result;
    	
    	result = new ArrayList<>();
    	result.add(new DefaultConverterEntry<>(BigDecimal.class, BigDecimalConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Boolean.class, BooleanConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Byte.class, ByteConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Character.class, CharacterConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Date.class, DateConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Double.class, DoubleConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Enum.class, EnumConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Float.class, FloatConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Integer.class, IntegerConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Long.class, LongConverter.class, true));
    	result.add(new DefaultConverterEntry<>(Short.class, ShortConverter.class, true));
    	return result;
	}
	
    /**
     * Returns built-in adjusters, converters, and validators. 
     */
	public static List<PolicyEntry<?,?,?>> getBuiltInPolicies() {
    	List<PolicyEntry<?,?,?>> result;
    	
    	result = new ArrayList<>();
    	result.add(new PolicyEntry<>(BigDecimalConversion.class, BigDecimalConverter.class, BigDecimal.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(BooleanConversion.class, BooleanConverter.class, Boolean.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(ByteConversion.class, ByteConverter.class, Byte.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(CharacterConversion.class, CharacterConverter.class, Character.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(DateConversion.class, DateConverter.class, Date.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(DoubleConversion.class, DoubleConverter.class, Double.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(EnumConversion.class, EnumConverter.class, Enum.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(FloatConversion.class, FloatConverter.class, Float.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(IntegerConversion.class, IntegerConverter.class, Integer.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(IntegerCSVConversion.class, IntegerCSVConverter.class, Integer.class, AnnotationUsage.COLLECTION_CONVERT, true, true));
    	result.add(new PolicyEntry<>(IntegerRange.class, IntegerRangeValidator.class, Integer.class, AnnotationUsage.POST_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(LongConversion.class, LongConverter.class, Long.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(MaxLength.class, MaxLengthValidator.class, null, AnnotationUsage.NON_CONVERT_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(MinInteger.class, MinIntegerValidator.class, Integer.class, AnnotationUsage.POST_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(Regex.class, RegexValidator.class, null, AnnotationUsage.NON_CONVERT_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(Required.class, RequiredValidator.class, null, AnnotationUsage.NON_CONVERT_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(RequiredIntegerEntries.class, RequiredIntegerEntriesValidator.class, Integer.class, AnnotationUsage.COLLECTION_POST_VALIDATION, true, false));
    	result.add(new PolicyEntry<>(ShortConversion.class, ShortConverter.class, Short.class, AnnotationUsage.CONVERT, true, true));
    	result.add(new PolicyEntry<>(StringCSVConversion.class, StringCSVConverter.class, String.class, AnnotationUsage.COLLECTION_CONVERT, true, true));
    	result.add(new PolicyEntry<>(ToEndOfDay.class, ToEndOfDayAdjuster.class, Date.class, AnnotationUsage.POST_ADJUSTER, true, false));
    	result.add(new PolicyEntry<>(ToLowerCase.class, ToLowerCaseAdjuster.class, null, AnnotationUsage.ADJUSTER, true, false));
    	result.add(new PolicyEntry<>(ToStartOfDay.class, ToStartOfDayAdjuster.class, Date.class, AnnotationUsage.POST_ADJUSTER, true, false));
    	result.add(new PolicyEntry<>(ToUpperCase.class, ToUpperCaseAdjuster.class, null, AnnotationUsage.ADJUSTER, true, false));
    	result.add(new PolicyEntry<>(Trim.class, TrimAdjuster.class, null, AnnotationUsage.ADJUSTER, true, false));
    	return result;
	}
	
    /**
     * Returns configured policy instance that processes an annotated form field, or result of NA type if not recognised. 
     */
    @SuppressWarnings("unchecked")
    public static <T> ConfiguredPolicy<T> getConfiguredPolicy(Annotation annotation, PolicyLookup policyLookup) throws Exception {
        Adjuster<Annotation> adjuster;
        ConfiguredPolicy<T> result;
        CollectionConverter<Annotation,T> collectionConverter;
        CollectionPostConversionAdjuster<Annotation,T> collectionPostConversionAdjuster;
        CollectionPostConversionValidator<Annotation,T> collectionPostConversionValidator;
        Constructor<?> constructor;
        Converter<Annotation,T> converter;
        CustomAdjuster customAdjusterAnnotation;
        CustomCollectionConversion customCollectionConversionAnnotation;
        CustomCollectionPostConversionAdjuster customCollectionPostConversionAdjusterAnnotation;
        CustomCollectionPostConversionValidation customCollectionPostConversionValidationAnnotation;
        CustomConversion customConversionAnnotation;
        CustomPostConversionAdjuster customPostConversionAdjusterAnnotation;
        CustomPostConversionValidation customPostConversionValidationAnnotation;
        CustomValidation customValidationAnnotation;
        NonConversionValidator<Annotation> nonConversionValidator;
        PolicyEntry<?,?,?> policyEntry;
        PostConversionAdjuster<Annotation,T> postConversionAdjuster;
        PostConversionValidator<Annotation,T> postConversionValidator;
        Policy<Annotation> policy;
        Class<?> policyClass;
        
        if (annotation instanceof CustomAdjuster) {
            customAdjusterAnnotation = (CustomAdjuster)annotation;
            policyClass = customAdjusterAnnotation.adjusterClass();
            if (Adjuster.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                adjuster = (Adjuster<Annotation>)constructor.newInstance(new Object[] {});
                adjuster.setAnnotation(customAdjusterAnnotation);
                result = ConfiguredPolicy.makeAdjusterResult(adjuster);
            } else {
                LOG.warn("CustomAdjuster annotation refers to class that doesn't implement Adjuster" + 
                        "  annotation=" + annotation.annotationType() + 
                        "  adjuster class=" + ((CustomAdjuster) annotation).adjusterClass());
                result = null;
            }
        } else if (annotation instanceof CustomConversion) {
            customConversionAnnotation = (CustomConversion)annotation;
            policyClass = customConversionAnnotation.validatorClass();
            if (Converter.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                converter = (Converter<Annotation,T>)constructor.newInstance(new Object[] {});
                converter.setAnnotation(customConversionAnnotation);
                result = ConfiguredPolicy.makeConverterResult(converter, false);
            } else {
                LOG.warn("CustomConversion annotation refers to class that doesn't implement Converter" + 
                        "  annotation=" + annotation.annotationType() + 
                        "  converter class=" + ((CustomConversion) annotation).validatorClass());
                result = null;
            }
        } else if (annotation instanceof CustomPostConversionAdjuster) {
            customPostConversionAdjusterAnnotation = (CustomPostConversionAdjuster)annotation;
            policyClass = customPostConversionAdjusterAnnotation.adjusterClass();
            if (PostConversionAdjuster.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                postConversionAdjuster = (PostConversionAdjuster<Annotation,T>)constructor.newInstance(new Object[] {});
                postConversionAdjuster.setAnnotation(customPostConversionAdjusterAnnotation);
                result = ConfiguredPolicy.makePostConversionAdjusterResult(postConversionAdjuster);
            } else {
                LOG.warn("CustomPostConversionAdjuster annotation refers to class that doesn't implement PostConversionAdjuster" + 
                            "  annotation=" + annotation.annotationType() + 
                            "  converter class=" + ((CustomPostConversionAdjuster) annotation).adjusterClass());
                    result = null;
            }
        } else if (annotation instanceof CustomPostConversionValidation) {
            customPostConversionValidationAnnotation = (CustomPostConversionValidation)annotation;
            policyClass = customPostConversionValidationAnnotation.validatorClass();
            if (PostConversionValidator.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                postConversionValidator = (PostConversionValidator<Annotation,T>)constructor.newInstance(new Object[] {});
                postConversionValidator.setAnnotation(customPostConversionValidationAnnotation);
                result = ConfiguredPolicy.makePostConversionValidatorResult(postConversionValidator);
            } else {
            	LOG.warn("CustomPostConversionValidation annotation refers to class that doesn't implement PostConversionValidator" + 
                            "  annotation=" + annotation.annotationType() + 
                            "  converter class=" + ((CustomPostConversionValidation) annotation).validatorClass());
                result = null;
            }
        } else if (annotation instanceof CustomValidation) {
            customValidationAnnotation = (CustomValidation)annotation;
            policyClass = customValidationAnnotation.validatorClass();
            if (NonConversionValidator.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                nonConversionValidator = (NonConversionValidator<Annotation>)constructor.newInstance(new Object[] {});
                nonConversionValidator.setAnnotation(customValidationAnnotation);
                result = ConfiguredPolicy.makeNonConversionValidatorResult(nonConversionValidator);
            } else {
                LOG.warn("CustomValidation annotation refers to class that doesn't implement NonConversionValidator" + 
                        "  annotation=" + annotation.annotationType() + 
                        "  converter class=" + ((CustomValidation) annotation).validatorClass());
            result = null;
            }
        } else if (annotation instanceof CustomCollectionConversion) {
            customCollectionConversionAnnotation = (CustomCollectionConversion)annotation;
            policyClass = customCollectionConversionAnnotation.validatorClass();
            if (CollectionConverter.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                collectionConverter = (CollectionConverter<Annotation,T>)constructor.newInstance(new Object[] {});
                collectionConverter.setAnnotation(customCollectionConversionAnnotation);
                result = ConfiguredPolicy.makeCollectionConverterResult(collectionConverter, false);
            } else {
                LOG.warn("CustomCollectionConversion annotation refers to class that doesn't implement CollectionConverter" + 
                        "  annotation=" + annotation.annotationType() + 
                        "  converter class=" + ((CustomConversion) annotation).validatorClass());
                result = null;
            }
        } else if (annotation instanceof CustomCollectionPostConversionAdjuster) {
            customCollectionPostConversionAdjusterAnnotation = (CustomCollectionPostConversionAdjuster)annotation;
            policyClass = customCollectionPostConversionAdjusterAnnotation.adjusterClass();
            if (CollectionPostConversionAdjuster.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                collectionPostConversionAdjuster = (CollectionPostConversionAdjuster<Annotation,T>)constructor.newInstance(new Object[] {});
                collectionPostConversionAdjuster.setAnnotation(customCollectionPostConversionAdjusterAnnotation);
                result = ConfiguredPolicy.makeCollectionPostConversionAdjusterResult(collectionPostConversionAdjuster);
            } else {
                LOG.warn("CustomPostConversionAdjuster annotation refers to class that doesn't implement PostConversionAdjuster" + 
                            "  annotation=" + annotation.annotationType() + 
                            "  converter class=" + ((CustomPostConversionAdjuster) annotation).adjusterClass());
                    result = null;
            }
        } else if (annotation instanceof CustomCollectionPostConversionValidation) {
            customCollectionPostConversionValidationAnnotation = (CustomCollectionPostConversionValidation)annotation;
            policyClass = customCollectionPostConversionValidationAnnotation.validatorClass();
            if (CollectionPostConversionValidator.class.isAssignableFrom(policyClass)) {
                constructor = policyClass.getConstructor();
                collectionPostConversionValidator = (CollectionPostConversionValidator<Annotation,T>)constructor.newInstance(new Object[] {});
                collectionPostConversionValidator.setAnnotation(customCollectionPostConversionValidationAnnotation);
                result = ConfiguredPolicy.makeCollectionPostConversionValidatorResult(collectionPostConversionValidator);
            } else {
                LOG.warn("CustomCollectionPostConversionValidation annotation refers to class that doesn't implement CollectionPostConversionValidator" + 
                            "  annotation=" + annotation.annotationType() + 
                            "  converter class=" + ((CustomConversion) annotation).validatorClass());
                    result = null;
            }
        } else if (annotation instanceof FormField) {
            result = ConfiguredPolicy.makeFormFieldResult(annotation);
        } else if (annotation instanceof ManualParameterConversion) {
            result = ConfiguredPolicy.makeManualParameterConversionResult(annotation);
        } else {
            result = null;
            policyEntry = policyLookup.getPolicyEntry(annotation.annotationType());
            if (policyEntry != null) {
	           	policyClass = policyEntry.getPolicyClass();
	            constructor = policyClass.getConstructor();
	            policy = (Policy<Annotation>)constructor.newInstance(new Object[] {});
	            policy.setAnnotation(annotation);
	            switch (policyEntry.getAnnotationUsage()) {
				case ADJUSTER:
	                result = ConfiguredPolicy.makeAdjusterResult((Adjuster<Annotation>)policy);
					break;
				case COLLECTION_CONVERT:
	                result = ConfiguredPolicy.makeCollectionConverterResult((CollectionConverter<Annotation,T>)policy, false);
					break;
				case COLLECTION_POST_ADJUSTER:
	                result = ConfiguredPolicy.makeCollectionPostConversionAdjusterResult((CollectionPostConversionAdjuster<Annotation,T>)policy);
					break;
				case COLLECTION_POST_VALIDATION:
	                result = ConfiguredPolicy.makeCollectionPostConversionValidatorResult((CollectionPostConversionValidator<Annotation,T>)policy);
					break;
				case CONVERT:
	                result = ConfiguredPolicy.makeConverterResult((Converter<Annotation,T>)policy, false);
					break;
				case FORM_FIELD:
				case MANUAL_PARAMETER_CONVERSION:
				case NA:
					// Not policies or checked above
					break;
				case NON_CONVERT_VALIDATION:
	                result = ConfiguredPolicy.makeNonConversionValidatorResult((NonConversionValidator<Annotation>)policy);
					break;
				case POST_ADJUSTER:
	                result = ConfiguredPolicy.makePostConversionAdjusterResult((PostConversionAdjuster<Annotation,T>)policy);
					break;
				case POST_VALIDATION:
	                result = ConfiguredPolicy.makePostConversionValidatorResult((PostConversionValidator<Annotation,T>)policy);
					break;
	            }
            } else {
            	LOG.debug("Annotation noticed by form field validation but not linked to any policy, which may be a mistake" + 
                    "  annotation=" + annotation.annotationType());
            }
            if (result == null) {
                result = ConfiguredPolicy.makeNAResult(annotation);
            }
        }
        
        return result;
    }

	/**
	 * Returns class used in default converter and default collection converter lookup from a form field's type or item 
	 * class for collection types. 
	 */
	public static Class<?> getConverterClassLookupFromFormFieldClass(Class<?> fieldClass) {
		Class<?> result;
		
		result = fieldClass;
	    if (boolean.class.isAssignableFrom(fieldClass)) {
	    	result = Boolean.class;
        } else if (byte.class.isAssignableFrom(fieldClass)) {
	    	result = Byte.class;
        } else if (char.class.isAssignableFrom(fieldClass)) {
	    	result = Character.class;
	    } else if (double.class.isAssignableFrom(fieldClass)) {
	    	result = Double.class;
	    } else if (Enum.class.isAssignableFrom(fieldClass)) {
	    	result = Enum.class;
        } else if (float.class.isAssignableFrom(fieldClass)) {
	    	result = Float.class;
        } else if (int.class.isAssignableFrom(fieldClass)) {
	    	result = Integer.class;
        } else if (long.class.isAssignableFrom(fieldClass)) {
	    	result = Long.class;
        } else if (short.class.isAssignableFrom(fieldClass)) {
	    	result = Short.class;
        }
	    return result;
	}

	/**
	 * Returns default collection converter for a collection form field or display field, or null if none available. 
	 */
	public static CollectionConverter<?,?> getDefaultCollectionConverter(Field field, PolicyLookup policyLookup) {
		Class<?> itemClass, collectionConverterClass;
		DefaultCollectionConverterEntry<?,?> collectionConverterEntry;
		CollectionConverter<?,?> result;
		
		result =  null;
		itemClass = getTypeFromCollectionField(field);
		collectionConverterEntry = policyLookup.getDefaultCollectionConverterEntry(itemClass);
		if (collectionConverterEntry != null) {
			collectionConverterClass = collectionConverterEntry.getCollectionConverterClass();
	    	try {
	    		result = (CollectionConverter<?,?>)collectionConverterClass.newInstance();
			}
	    	catch (Exception e) {
				LOG.error("Creation of collection  converter failed  class=" + collectionConverterClass, e);
				result =  null;
			}
		}
    	return result;
	}
	
	/**
	 * Returns default converter for a single value form field or display field, or null if none available. 
	 */
    public static Converter<?,?> getDefaultConverter(Field field, PolicyLookup policyLookup) {
		Class<?> fieldClass, converterClass;
		DefaultConverterEntry<?,?> converterEntry;
		Converter<?,?> result;
		
		result =  null;
		fieldClass = field.getType();
		converterEntry = policyLookup.getDefaultConverterEntry(fieldClass);
		if (converterEntry != null) {
    		converterClass = converterEntry.getConverterClass();
	    	try {
	    		result = (Converter<?,?>)converterClass.newInstance();
			}
	    	catch (Exception e) {
				LOG.error("Creation of converter failed  class=" + converterClass, e);
				result =  null;
			}
		}
    	return result;
	}
    
	/**
	 * Returns default converter for each item of an array display field, or null if none available.  Not used for 
	 * form fields. 
	 */
	public static Converter<?,?> getDefaultConverterFromArray(Field field, PolicyLookup policyLookup) {
		Class<?> itemClass, converterClass;
		DefaultConverterEntry<?,?> converterEntry;
		Converter<?,?> result;
		
		result =  null;
		itemClass = getTypeFromArrayField(field);
		converterEntry = policyLookup.getDefaultConverterEntry(itemClass);
		if (converterEntry != null) {
			converterClass = converterEntry.getConverterClass();
	    	try {
	    		result = (Converter<?,?>)converterClass.newInstance();
			}
	    	catch (Exception e) {
				LOG.error("Creation of converter failed  class=" + converterClass, e);
				result =  null;
			}
		}
    	return result;
	}
	
	/**
	 * Returns default converter for each item of a collection display field, or null if none available.  Not used for 
	 * form fields. 
	 */
	public static Converter<?,?> getDefaultConverterFromCollection(Field field, PolicyLookup policyLookup) {
		Class<?> itemClass, converterClass;
		DefaultConverterEntry<?,?> converterEntry;
		Converter<?,?> result;
		
		result =  null;
		itemClass = getTypeFromCollectionField(field);
		converterEntry = policyLookup.getDefaultConverterEntry(itemClass);
		if (converterEntry != null) {
			converterClass = converterEntry.getConverterClass();
	    	try {
	    		result = (Converter<?,?>)converterClass.newInstance();
			}
	    	catch (Exception e) {
				LOG.error("Creation of converter failed  class=" + converterClass, e);
				result =  null;
			}
		}
    	return result;
	}
	
    /**
     * Returns the default field name to receive a converted form field.  
     */
    public static String getDefaultRecipientName(String fieldName) {
        return "parsed" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
    }
    
    /**
     * Returns the object expected to receive submitted form data on behalf of an Action.  This can be the model of a 
     * ModelDriven Action, or the Action itself.   
     */
    public static Object getForm(Object action) {
        boolean modelDriven;
        
        modelDriven = action instanceof ModelDriven;
        if (modelDriven) {
            return ((ModelDriven<?>)action).getModel();
        } else {
            return action;
        }
    }

    public static FormFieldCategory getFormFieldCategory(FieldUsage<?> fieldUsage) {
        if (fieldUsage.getFieldType() == null) {
            return FormFieldCategory.MANUAL_CONVERSION;
        } else if (File.class.isAssignableFrom(fieldUsage.getFieldType())) {
    		return FormFieldCategory.FILE;
    	} else if (fieldUsage.getArray() && String.class.isAssignableFrom(fieldUsage.getFieldType())) {
    		return FormFieldCategory.STRING_ARRAY;
    	} else if (fieldUsage.getArray()) {
    		return FormFieldCategory.NON_STRING_ARRAY;
    	} else if (fieldUsage.getCollection() && String.class.isAssignableFrom(fieldUsage.getFieldType())) {
    		return FormFieldCategory.STRING_COLLECTION;
    	} else if (fieldUsage.getCollection()) {
    		return FormFieldCategory.NON_STRING_COLLECTION;
    	} else if (String.class.isAssignableFrom(fieldUsage.getFieldType())) {
    		return FormFieldCategory.STRING;
    	} else {
    		return FormFieldCategory.NON_STRING;
    	}
    }
    
	/**
     * Returns what policies should apply to a form field, such as conversion and validation, or throws 
     * {@link FormFieldAnnotationTypeMismatchException} if any annotation does not match the field's type. 
     */
    @SuppressWarnings("unchecked")
	public static <T> FieldUsage<T> getFormFieldUsage(Field formField, PolicyLookup policyLookup) throws FormFieldAnnotationTypeMismatchException {
    	ConfiguredPolicy<T> converterConfiguredPolicy;
    	AnnotationEntries<T> annotationEntries;
    	FieldUsage<T> result;
    	Class<T> fieldType;
    	boolean isArray, isCollection;
        
    	annotationEntries = getAnnotationEntries(formField, policyLookup);
    	isArray = formField.getType().isArray();
    	isCollection = Collection.class.isAssignableFrom(formField.getType());
    	if (isArray) {
    		fieldType = (Class<T>)formField.getType().getComponentType();
    	} else if (isCollection) {
    		fieldType = getTypeFromCollectionField(formField);
    	} else {
    		fieldType = (Class<T>)formField.getType();
    	}
    	if (isArray) {
	    	// Annotations apply to single value and collections but not arrays
    	} else if (isCollection) {
        	// Check all annotations are apply to field's type
    		converterConfiguredPolicy = annotationEntries.getCollectionConverter();
    		if (converterConfiguredPolicy != null) {
	    		if (fieldType == null) {
	    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
	    		}
	    		if (!checkFieldClass(converterConfiguredPolicy.getCollectionConverter().getRecipientClass(), fieldType)) {
	    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
	    		}
		    	for (ConfiguredPolicy<?> configuredPolicy: annotationEntries.getCollectionPostConversionValidators()) {
		    		if (!checkFieldClass(configuredPolicy.getCollectionPostConversionValidator().getRecipientClass(), fieldType)) {
		    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
		    		}
		    	}
    		}
	    } else if (String.class.isAssignableFrom(formField.getType())) {
    		// Any converters and post-conversion validators on a string field apply to the other half of a 
	    	// formatted/unformatted pair, which is checked for compatibility later
    	} else {
        	// Check all annotations are apply to field's type
            converterConfiguredPolicy = annotationEntries.getConverter();
    		if (converterConfiguredPolicy != null) {
	    		if (fieldType == null) {
	    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
	    		}
	    		if (!checkFieldClass(converterConfiguredPolicy.getConverter().getRecipientClass(), fieldType)) {
	    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
	    		}
		    	for (ConfiguredPolicy<?> configuredPolicy: annotationEntries.getPostConversionValidators()) {
		    		if (!checkFieldClass(configuredPolicy.getPostConversionValidator().getRecipientClass(), fieldType)) {
		    			throw new FormFieldAnnotationTypeMismatchException(converterConfiguredPolicy.getAnnotation(), formField);
		    		}
		    	}
    		}
    	}
    	
    	result = new FieldUsage<T>();
    	result.setField(formField);
    	result.setName(formField.getName());
    	result.setAnnountationEntries(annotationEntries);
    	result.setFieldType(fieldType);
    	result.setArray(isArray);
    	result.setCollection(isCollection);
    	return result;
    }
    
    /**
     * Returns all properties of a class, even private ones, whether directly declared or inherited. 
     */
    public static Collection<Field> getProperties(Class<?> type) {
        Collection<Field> fields;
        Class<?> c;
        Field[] declaredfields;
        int modifiers;
        
        fields = new ArrayList<Field>();
        for (c = type; c != null; c = c.getSuperclass()) {
        	declaredfields = c.getDeclaredFields();
        	for (Field declaredfield: declaredfields) {
        		modifiers = declaredfield.getModifiers();
        		if (!Modifier.isStatic(modifiers)) {
        			fields.add(declaredfield);
        		}
        	}
        }
        return fields;
    }
    
    /**
     * Returns property of a class, even if private or inherited, or null if not found. 
     */
    public static Field getProperty(Class<?> type, String name) {
        Class<?> c;
        Field[] declaredFields;
        
        for (c = type; c != null; c = c.getSuperclass()) {
        	declaredFields = c.getDeclaredFields();
        	for (Field f: declaredFields) {
        		if (f.getName().equals(name)) {
        			return f;
        		}
        	}
        }
        return null;
    }
    
    /**
     * Returns a processed form stored in session.
     */
    public static StoredForm getStoredForm() {
    	Map<String, Object> session;
        Object rawObject;
        
        session = ActionContext.getContext().getSession();
        if (session != null) {
	        rawObject = session.get(FormStoreInterceptor.SESSION_STORED_FORM);
	        if (rawObject != null) {
	            if (rawObject instanceof StoredForm) {
	                return (StoredForm)rawObject;
	            } else {
	                LOG.warn("Session attribute is not a stored form  session attribute=" + FormStoreInterceptor.SESSION_STORED_FORM +
	                    "  value=" + rawObject);
	            }
	        }
        }
        return null;
    }
    
	/**
     * Returns element type of a collection-based form field. 
     */
    @SuppressWarnings("unchecked")
    public static <T> Class<T> getTypeFromArrayField(Field recipientField) {
        Class<T> recipientFieldClass;
        
        if (!recipientField.getType().isArray()) {
            return null;
        }
        recipientFieldClass = (Class<T>)recipientField.getType().getComponentType();
       	return recipientFieldClass; 
    }
    
	/**
     * Returns element type of a collection-based form field. 
     */
    @SuppressWarnings("unchecked")
    public static <T> Class<T> getTypeFromCollectionField(Field recipientField) {
        Class<?> recipientFieldClass;
        Type[] recipientEntryTypes;
        
        recipientFieldClass = recipientField.getType();
        if (!Collection.class.isAssignableFrom(recipientFieldClass)) {
            return null;
        }
        recipientEntryTypes = ((ParameterizedType)recipientField.getGenericType()).getActualTypeArguments();
        if (recipientEntryTypes.length != 1) {
            return null;
        }
        if (recipientEntryTypes[0] instanceof Class) {
        	return (Class<T>)recipientEntryTypes[0];
        } else if (recipientEntryTypes[0] instanceof ParameterizedType) {
        	return (Class<T>)((ParameterizedType)recipientEntryTypes[0]).getRawType();
        } else {
        	return null;
        }
    }
    
    /**
     * Returns whether an annotation class is for custom policies.
     */
    public static <A extends Annotation> boolean isCustomAnnotationClass(Class<A> annotationClass) {
    	return customAnnotations.contains(annotationClass);
    }

	/**
     * Returns an empty, modifiable collection from a collection type. 
     */
	public static <T> Collection<T> makeCollectionForRecipient(Class<?> recipientClass) {
		if (BlockingQueue.class.isAssignableFrom(recipientClass)) {
            return new LinkedBlockingQueue<>();
        } else if (Deque.class.isAssignableFrom(recipientClass)) {
            return new LinkedList<>();
        } else if (List.class.isAssignableFrom(recipientClass)) {
            return new ArrayList<>();
        } else if (NavigableSet.class.isAssignableFrom(recipientClass)) {
            return new TreeSet<>();
        } else if (Queue.class.isAssignableFrom(recipientClass)) {
            return new LinkedList<>();
        } else if (SortedSet.class.isAssignableFrom(recipientClass)) {
            return new TreeSet<>();
        } else if (Set.class.isAssignableFrom(recipientClass)) {
            return new HashSet<>();
        } else if (TransferQueue.class.isAssignableFrom(recipientClass)) {
            return new LinkedTransferQueue<>();
        } else {
            return new ArrayList<>();
        }
    }

}