package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;

import javax.servlet.http.HttpServletRequest;

import name.matthewgreet.strutscommons.action.AbstractSaveListActionSupport.SaveResponse.ListAction;
import name.matthewgreet.strutscommons.form.AbstractForm;
import name.matthewgreet.strutscommons.util.AbstractCompositeCache;
import name.matthewgreet.strutscommons.util.ListCache;




/**
 * <P>Template class for changing business data related to the currently, selected item.</P>
 * 
 * <P>Concrete subclasses must implement {@link #getSaveListConfig}, which defines the the 
 * {@link AbstractCompositeCache} being used.  Subclasses must override {@link #doSave}.</P>
 * 
 * <P>The URL parameters, whose names can be changed by config, that can be used are:-</P>
 * 
 * <DL>
 *   <DT>select</DT><DD>Index of item in list cache to update (which starts at 0).</DD>
 * </DL>
 * 
 * <DL>
 *   <DT>M</DT><DD>Record type of master list cache if loading a slave list or Object otherwise.</DD>
 *   <DT>K</DT><DD>Primary key type of list cache.</DD>
 *   <DT>T</DT><DD>Record type of list cache.</DD>
 *   <DT>F</DT><DD>Type of Struts action form used by this action or NullForm if Struts forms aren't used.</DD>
 * </DL>
 */
public abstract class AbstractSaveListActionSupport<M extends Serializable,K extends Serializable,T extends Serializable,F extends AbstractForm> extends AbstractFormDrivenActionSupport<F> {
    private static final long serialVersionUID = 8213325828567959710L;
    
    // Name of parameter for choosing record of list to display
    public static final String PARAMETER_NAME_INDEXSELECT = "select";
    
    /**
     * Configures list that is updated by save action.  Configuration is fixed 
     * and does not change during the action's lifetime.
     */
    public static class SaveListConfig {
        private String errorForwardName;
        private String parameterNameIndexSelect;
        
        public SaveListConfig() {
            errorForwardName = null;
            parameterNameIndexSelect = PARAMETER_NAME_INDEXSELECT;
        }
        
        public SaveListConfig(String errorForwardName) {
            this.errorForwardName = errorForwardName;
            parameterNameIndexSelect = PARAMETER_NAME_INDEXSELECT;
        }
        
        /**
         * Written by subclasses to return name of Struts forward (usually 
         * global forward) to be used for unhandlable errors.  Defaults to 
         * 'error'.
         */
        public String getErrorForwardName() {
            return errorForwardName;
        }
        public void setErrorForwardName(String value) {
            errorForwardName = value;
        }
        
        /**
         * Returns name of request parameter that specifies selected item by its
         * index number (starting at 0).  Defaults to 'select'.   
         */
        public String getParameterNameIndexSelect() {
            return parameterNameIndexSelect;
        }
        public void setParameterNameIndexSelect(String value) {
            parameterNameIndexSelect = value;
        }

    }
    
    /**
     * Forwarding and cache updating response set by concrete subclass.
     * <P>If succeeded is set, action will forward or redirect to the success 
     * page, otherwise forwards to input page (as defined in Struts action 
     * mapping).  If succeeded is set, the selected item of the cache is 
     * replaced from the updatedItem property unless it is null, which will mark 
     * the list for reload.  The success page is derived from the action mapping 
     * using the successForwardName property, which defaults to 'success', 
     * unless the successForwardURL is set.  Error messages should included if 
     * failed.<P>        
     */
    public static class SaveResponse<T> {
        /**
         * <DL>
         * <DT>ADD_AND_RELOAD_ITEM</DT><DD>Adds item to list and reloads item for all required data.</DD>
         * <DT>ADD_ITEM</DT><DD>Adds item to list, which must have all expected data.</DD>
         * <DT>DESELECT</DT><DD>Sets the list selection to no item.</DD>
         * <DT>NONE</DT><DD>Does not affect list.</DD>
         * <DT>RELOAD_ENTIRE</DT><DD>Reloads master list, which means all slave lists are cleared for reload.</DD>
         * <DT>RELOAD_ITEM</DT><DD>Reloads selected item of list only.</DD>
         * <DT>RELOAD_LIST</DT><DD>Reloads list using current list finder.</DD>
         * <DT>RELOAD_MASTER</DT><DD>Reloads selected item of master list, which means all slave lists are cleared for reload.</DD>
         * <DT>REMOVE_ITEM</DT><DD>Removes selected item of list.</DD>
         * </DL> 
         */
        public enum ListAction {ADD_AND_RELOAD_ITEM, ADD_ITEM, DESELECT, NONE, RELOAD_ENTIRE, RELOAD_ITEM, RELOAD_LIST, RELOAD_MASTER, REMOVE_ITEM}
        
        /**
         * Returns response for when an item is successfully created, complete with other list data. 
         */
        public static <T> SaveResponse<T> makeCreatedItemSuccessResponse(T createdItem, String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.ADD_ITEM);
            result.setSucceeded(true);
            result.setUpdatedItem(createdItem);
            result.setUpdateItem(true);
            return result;
        }
        
        /**
         * Returns response for when an item is successfully created, complete with other list data. 
         */
        public static <T> SaveResponse<T> makeCreatedAndReloadItemSuccessResponse(T createdItem, String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.ADD_AND_RELOAD_ITEM);
            result.setSucceeded(true);
            result.setUpdatedItem(createdItem);
            result.setUpdateItem(true);
            return result;
        }
        
        /**
         * Returns response for changing the list selection to no item.  
         */
        public static <T> SaveResponse<T> makeDeselectResponse() {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.setListAction(ListAction.DESELECT);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action fails. 
         */
        public static <T> SaveResponse<T> makeFailureResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addErrorMessage(message);
            result.setListAction(ListAction.NONE);
            result.setSucceeded(false);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action fails but uses any existing error messages. 
         */
        public static <T> SaveResponse<T> makeFailureResponseExistingMessages() {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.setListAction(ListAction.NONE);
            result.setSucceeded(false);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action fails and user needs to see a reloaded item to see why. 
         */
        public static <T> SaveResponse<T> makeFailureAndReloadItemResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addErrorMessage(message);
            result.setListAction(ListAction.RELOAD_ITEM);
            result.setSucceeded(false);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action fails and user needs to see a reloaded list to see why. 
         */
        public static <T> SaveResponse<T> makeFailureAndReloadListResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addErrorMessage(message);
            result.setListAction(ListAction.RELOAD_LIST);
            result.setSucceeded(false);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action fails and user needs to see a reloaded master record and child list to 
         * see why. 
         */
        public static <T> SaveResponse<T> makeFailureAndReloadMasterResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addErrorMessage(message);
            result.setListAction(ListAction.RELOAD_MASTER);
            result.setSucceeded(false);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when an action is successful but does not change the selected item. 
         */
        public static <T> SaveResponse<T> makeInvisibleSuccessResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.NONE);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when the selected item is successfully removed. 
         */
        public static <T> SaveResponse<T> makeRemovedItemSuccessResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.REMOVE_ITEM);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        public static <T> SaveResponse<T> makeSuccessAndReloadItemResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.RELOAD_ITEM);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        public static <T> SaveResponse<T> makeSuccessAndReloadListResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.RELOAD_LIST);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        public static <T> SaveResponse<T> makeSuccessAndReloadMasterResponse(String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.RELOAD_MASTER);
            result.setSucceeded(true);
            result.setUpdateItem(false);
            return result;
        }
        
        /**
         * Returns response for when the selected item is successfully updated. 
         */
        public static <T> SaveResponse<T> makeUpdatedItemSuccessResponse(T updatedItem, String message) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.addInfoMessage(message);
            result.setListAction(ListAction.NONE);
            result.setSucceeded(true);
            result.setUpdatedItem(updatedItem);
            result.setUpdateItem(true);
            return result;
        }
        
        /**
         * Returns response for when the selected item is successfully updated. 
         */
        public static <T> SaveResponse<T> makeUpdatedItemSuccessSilentResponse(T updatedItem) {
            SaveResponse<T> result;
            
            result = new SaveResponse<T>();
            result.setListAction(ListAction.NONE);
            result.setSucceeded(true);
            result.setUpdatedItem(updatedItem);
            result.setUpdateItem(true);
            return result;
        }
        
        private boolean succeeded = false;
        private String failureResultName = "input";
        private String successResultName = "success";
        private boolean updateItem = false;
        private T updatedItem = null;
        private ListAction listAction = ListAction.NONE;
        private Collection<String> errors = new ArrayList<String>();
        private Collection<String> messages = new ArrayList<String>();
        
        /**
         * Default constructor.
         */
        public SaveResponse() {
            // Empty
        }
        
        
        /**
         * Indicates the save succeeded.  If set, template base class uses action result named by 
         * {@link #getSuccessResultName}, otherwise uses action result named by {@link #getFailureResultName}.  Defaults 
         * to false. 
         */
        public boolean getSucceeded() {
            return succeeded;
        }
        public void setSucceeded(boolean value) {
            succeeded = value;
        }
        
        /**
         * Action result name to use if response reports failure.  Defaults to "input".
         */
        public String getFailureResultName() {
            return failureResultName;
        }
        public void setFailureResultName(String value) {
            failureResultName = value;
        }

        /**
         * Action result name to use if response reports success.  Defaults to "success".
         */
        public String getSuccessResultName() {
            return successResultName;
        }
        public void setSuccessResultName(String value) {
            successResultName = value;
        }

        /**
         * Whether selected item should be updated or new item added.  Only used if update succeeded and cache reload 
         * mode is NONE.  Defaults to false.
         */
        public boolean getUpdateItem() {
            return updateItem;
        }
        public void setUpdateItem(boolean value) {
            updateItem = value;
        }

        /**
         * Updated version of selected item in current cache list or item to add.
         * <P>This is typically returned by set business methods that update a 
         * record and returns the updated version.  Only used if update 
         * succeeded and cache reload mode is RELOAD_NONE or RELOAD_ADD.  
         * Defaults to null.</P>
         */
        public T getUpdatedItem() {
            return updatedItem;
        }
        public void setUpdatedItem(T value) {
            updatedItem = value;
        }

        /**
         * Cache list reload mode.
         * <P>See RELOAD_ constants.</P>
         */
        public ListAction getListAction() {
            return listAction;
        }
        public void setListAction(ListAction value) {
            listAction = value;
        }
        
        /**
         * Struts errors messages to be used, usually for update failure.  An error message does not indicate an error 
         * (use {@link #setSucceeded}) and is displayed even for success (though that's unusual). 
         */
        public Collection<String> getErrors() {
            return errors;
        }
        
        /**
         * Struts information messages to be used, usually for update success.  An info message does not indicate a 
         * success (use {@link #setSucceeded}) and is displayed even for failure (though that's unusual).  
         */
        public Collection<String> getMessages() {
            return messages;
        }
        
        /**
         * Convenience function to add error message.
         */
        public void addErrorMessage(String text) {
            errors.add(text);
        }
        
        /**
         * Convenience function to add info message.
         */
        public void addInfoMessage(String text) {
            messages.add(text);
        }

    }
    

    private final SaveListConfig config = getSaveListConfig();
    
    public AbstractSaveListActionSupport() {
        super();
    }
    
    /**
     * Return index of record to select according to request (if any).
     */
    private Integer checkIndexSelect(HttpServletRequest request) {
        String select;
        
        select = request.getParameter(config.getParameterNameIndexSelect());
        if (select != null) {
            try {
                return Integer.parseInt(select);
            }
            catch (NumberFormatException e) {
                // Ignore 
            	getLogger().warn("Value of select parameter is not a number  select=" + select);
            }
        }
        return null;
    }
    
    /**
     */
    @Override
    public String execute() throws Exception {
        ListCache<?,?,M> masterListCache;
        ListCache<M,K,T> listCache;
        SaveResponse<T> saveResponse;
        HttpServletRequest request;
        ListAction listAction;
        M selectedMaster;
        T updatedItem;
        Integer indexSelect;
        String forward;
        int newIndex, listSize;
        
        request = getServletRequest();
        masterListCache = getMasterListCache();
        listCache = getListCache();
        saveResponse = null;
        
        // Select different record if requested
        listSize = listCache.getListSize();
        indexSelect = checkIndexSelect(request);
        if (indexSelect == null) {
            indexSelect = getAlternateIndex(listCache);
        }
        if (hasErrors()) {    // getAlternateIndex can write errors
            return "input";
        }
        if (indexSelect != null) {
            newIndex = indexSelect;
            // Make sure index is within list boundary or -1.
            if (newIndex >= listSize) {
                newIndex = listSize - 1;
            } else if (newIndex < 0) {
                newIndex = -1;
            }
            listCache.setSelectedIndex(newIndex);
        }
        
        if (masterListCache != null && masterListCache != listCache) {
            selectedMaster = masterListCache.getSelected();
        } else {
            selectedMaster = null;
        }
        saveResponse = doSave(selectedMaster, listCache.getSelected());

        listAction = saveResponse.getListAction();
        updatedItem = saveResponse.getUpdatedItem();
        
        // Mark master list, list or selected item for reload as appropriate
        switch (listAction) {
        case ADD_AND_RELOAD_ITEM:
            listCache.addItem(updatedItem);
            if (listCache.getItemFinder() != null) {
                listCache.markSelectedReload();
            } else {
                listCache.markReload();
            }
            break;
        case ADD_ITEM:
            listCache.addItem(updatedItem);
            break;
        case DESELECT:
            listCache.setSelectedIndex(-1);
            break;
        case NONE:
            // Nothing
            break;
        case RELOAD_ITEM:
            if (listCache.getItemFinder() != null) {
                listCache.markSelectedReload();
            } else {
                listCache.markReload();
            }
            break;
        case RELOAD_LIST:
            listCache.markReload();
            break;
        case RELOAD_MASTER:
            if (masterListCache != null) {
                masterListCache.markSelectedReload();
            } else {    // If no master given, main list is master list
                listCache.markSelectedReload();
            }
            break;
        case RELOAD_ENTIRE:
            if (masterListCache != null) {
                masterListCache.markReload();
            } else {    // If no master given, main list is master list
                listCache.markReload();
            }
            break;
        case REMOVE_ITEM:
            listCache.removeSelected();
            break;
        }
        
        for (String message: saveResponse.getMessages()) {
            addActionMessage(message);
        }
        for (String errorMessage: saveResponse.getErrors()) {
            addActionError(errorMessage);
        }
        if (saveResponse.getSucceeded()) {
            if (saveResponse.getUpdateItem() && listAction == ListAction.NONE) {
                listCache.setSelected(updatedItem);
            }
            forward = saveResponse.getSuccessResultName();
        } else {
            forward = saveResponse.getFailureResultName();
        }
        return forward;
    }

    /**
     * Written by subclasses to try to save form data to business model.  The form data applies to the currently 
     * selected item displayed on the viewer page.  Returns response configuration, which includes whether the save 
     * succeeded and, if available, the updated item.
     * 
     * @param selectedItem Currently selected item in cache that will be changed, or null if none, generally for adding.
     * @return Response Configuration including success flag and any updated item.
     */
    protected abstract SaveResponse<T> doSave(M selectedMaster, T selectedItem)
        throws Exception;
    
    /**
     * Optionally written by subclasses to return index of record to select, typically by a request parameter, or null 
     * for no change of selection.  If any action or field errors are written, the action ends and the input result is 
     * used.
     */
    protected Integer getAlternateIndex(ListCache<M,K,T> listCache) {
        return null;
    }
    
    /**
     * Written by subclasses to return list cache that is being updated. 
     */
    protected abstract ListCache<M,K,T> getListCache();
 
    /**
     * Written by subclasses to return master list cache or null if the list cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache();
 
    /**
     * Written by subclasses to configure behaviour of view action. 
     */
    protected abstract SaveListConfig getSaveListConfig();
}
