package name.matthewgreet.strutscommons.action;

import java.util.Enumeration;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpSession;

import name.matthewgreet.strutscommons.interceptor.BrowserTabInterceptor2;

/**
 * <P>Like {@link HttpSession}, stores sets of named data or attribute about a user between requests but data are 
 * private to each browser tab, so different pages of the same browser don't clash when using the same name.  Instances
 * are injected into Actions implementing {@link BrowserTabAware2} by the {@link BrowserTabInterceptor2} interceptor.</P>
 * 
 * <P>User data, such as authentication, can be shared.  When an attribute, identified by its name, is read, it can come 
 * from a tab's private set or the shared set but not both.  The normal set function, setAttribute, sets a private 
 * attribute (unless it's already shared) whereas setSharedAttribute sets it as shared (also deleting private attributes
 * using the same name).  Actions of the same application are expected to know shared attribute names.</P>
 * 
 * <P>Implementations are wrappers around the request's session and shared attribute are stored in it, so such 
 * attributes can be accessed by code that don't recognise this class.</P>
 */
public interface BrowserTabSession extends HttpSession {
	/**
	 * Returns ids of all browser tabs known to exist for the user. 
	 */
	public Set<String> getAllTabIds();
	
	/**
	 * Returns object bound to name, whether as private or shared attribute, or null if none.
	 */
	public Object getAttribute(String name);
	
	/**
	 * Returns shallow copy of all attributes, whether private or shared.  Changing the map won't affect the session. 
	 */
	public Map<String, Object> getAttributeMap();
	
	/**
	 * Returns an enumeration of all attribute names, whether private or shared.
	 */
	public Enumeration<String> getAttributeNames();
	
	/**
	 * Returns underlying request session. 
	 */
	public HttpSession getHttpSession();
	
	/**
	 * Returns shallow copy of private, tab-specific attributes.  Changing the map won't affect the session.
	 */
	public Map<String, Object> getPrivateAttributeMap();
	
	/**
	 * Returns set of names of private, tab-specific attributes.
	 */
	public Set<String> getPrivateAttributeNames();
	
	/**
	 * Returns shallow copy of shared attributes.  Changing the map won't affect the session.
	 */
	public Map<String, Object> getSharedAttributeMap();
	
	/**
	 * Returns set of names of shared attributes.
	 */
	public Set<String> getSharedAttributeNames();
	
	/**
	 * Returns id of browser tab. 
	 */
	public String getTabId();
	
	/**
	 * Binds an object to a name as a private, tab-specific attribute, replacing any existing value.  If the attribute 
	 * already exists and is shared, the attribute remains shared.
	 */
	public void setAttribute(String name, Object value);
	
	/**
	 * Like {@link #setAttribute}, binds an object to a name but as a shared attribute.  If any private attributes have 
	 * the same name, they're removed.
	 */
	public void setSharedAttribute(String name, Object value);

}
