package name.matthewgreet.strutscommons.action;

import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionContext;

/**
 * Implementation of {@link BrowserTabSession}.
 */
@SuppressWarnings("deprecation")
public class BrowserTabSessionImpl implements BrowserTabSession {
	private static final String ATTR_NAME_PRIVATE_MAPS = BrowserTabSessionImpl.class + "_PRIVATE_MAPS";
	
	private HttpSession httpSession;
	private Map<String,Map<String,Object>> privateMaps;
	private String tabId;
	
	@SuppressWarnings("unchecked")
	public BrowserTabSessionImpl(HttpSession httpSession, String tabId) {
		this.httpSession = httpSession;
		this.tabId = tabId;
		privateMaps = (Map<String,Map<String,Object>>)httpSession.getAttribute(ATTR_NAME_PRIVATE_MAPS);
		if (privateMaps == null) {
			privateMaps = new HashMap<>();
			httpSession.setAttribute(ATTR_NAME_PRIVATE_MAPS, privateMaps);
		}
	}
	
	protected Object getAttributeInternal(String name) {
		Map<String,Object> privateMap;
		Object result;
		
		result = httpSession.getAttribute(name);
		if (result != null) {
			return result;
		}
		privateMap = privateMaps.get(tabId);
		if (privateMap == null) {
			return null;
		}
		result = privateMap.get(name);
		return result;
	}
	
	protected Map<String,Object> getGuaranteedPrivateMap() {
		Map<String,Object> result;
		
		result = privateMaps.get(tabId);
		if (result == null) {
			result = new HashMap<>();
			privateMaps.put(tabId, result);
		}
		return result;
	}

	protected Set<String> getPrivateAttributeNamesInternal() {
		Map<String,Object> privateMap;
		Set<String> result;
		
		privateMap = privateMaps.get(tabId);
		if (privateMap == null) {
			return new HashSet<>();
		}
		result = new HashSet<>(privateMap.keySet());
		return result;
	}

	protected Set<String> getSharedAttributeNamesInternal() {
		return new HashSet<String>(Collections.list(httpSession.getAttributeNames()));
	}
	
	protected void purgePrivateAttributes(String name) {
		for (Entry<String, Map<String, Object>> privateMapEntry: privateMaps.entrySet()) {
			privateMapEntry.getValue().remove(name);
		}
	}
	
	protected void removeAttributeInternal(String name) {
		Map<String,Object> privateMap;
		
		Object sharedObject;

		sharedObject = httpSession.getAttribute(name);
		if (sharedObject == null) {
			privateMap = privateMaps.get(tabId);
			if (privateMap != null) {
				privateMap.remove(name);
			}
		} else {
			httpSession.removeAttribute(name);
		}
	}

	protected void setAttributeInternal(String name, Object value) {
		Map<String,Object> privateMap;
		Object sharedObject;

		sharedObject = httpSession.getAttribute(name);
		if (sharedObject == null) {
			privateMap = getGuaranteedPrivateMap();
			privateMap.put(name, value);
		} else {
			httpSession.setAttribute(name, value);
		}
	}


	@Override
	public Set<String> getAllTabIds() {
		return new HashSet<>(privateMaps.keySet());
	}

	@Override
	public Object getAttribute(String name) {
		return getAttributeInternal(name);
	}

	@Override
	public Map<String, Object> getAttributeMap() {
		Map<String, Object> privateMap, result;
		
		privateMap = privateMaps.get(tabId);
		if (privateMap != null) {
			result = new HashMap<>(privateMap);
		} else {
			result = new HashMap<>();
		}
		for (String attributeName: Collections.list(httpSession.getAttributeNames())) {
			result.put(attributeName, httpSession.getAttribute(attributeName));
		}
		return result;
	}

	@Override
	public Enumeration<String> getAttributeNames() {
		Set<String> combinedAttributeNames;
		
		combinedAttributeNames = getSharedAttributeNamesInternal();
		combinedAttributeNames.addAll(getPrivateAttributeNamesInternal());
		return Collections.enumeration(combinedAttributeNames);
	}

	@Override
	public long getCreationTime() {
		return httpSession.getCreationTime();
	}

	@Override
	public HttpSession getHttpSession() {
		return httpSession;
	}

	@Override
	public String getId() {
		return httpSession.getId();
	}

	@Override
	public long getLastAccessedTime() {
		return httpSession.getLastAccessedTime();
	}

	@Override
	public int getMaxInactiveInterval() {
		return httpSession.getMaxInactiveInterval();
	}

	@Override
	public Map<String, Object> getPrivateAttributeMap() {
		Map<String, Object> privateMap, result;
		
		privateMap = privateMaps.get(tabId);
		if (privateMap != null) {
			result = new HashMap<>(privateMap);
		} else {
			result = new HashMap<>();
		}
		return result;
	}

	@Override
	public Set<String> getPrivateAttributeNames() {
		return getPrivateAttributeNamesInternal();
	}

	@Override
	public ServletContext getServletContext() {
		return httpSession.getServletContext();
	}

	@Override
	public HttpSessionContext getSessionContext() {
		return httpSession.getSessionContext();
	}

	@Override
	public Map<String, Object> getSharedAttributeMap() {
		Map<String, Object> result;
		
		result = new HashMap<>();
		for (String attributeName: Collections.list(httpSession.getAttributeNames())) {
			result.put(attributeName, httpSession.getAttribute(attributeName));
		}
		return result;
	}

	@Override
	public Set<String> getSharedAttributeNames() {
		return getSharedAttributeNamesInternal();
	}

	@Override
	public String getTabId() {
		return tabId;
	}

	@Override
	public Object getValue(String name) {
		return getAttributeInternal(name);
	}

	@Override
	public String[] getValueNames() {
		Set<String> combinedAttributeNames;
		
		combinedAttributeNames = getSharedAttributeNamesInternal();
		combinedAttributeNames.addAll(getPrivateAttributeNamesInternal());
		return combinedAttributeNames.toArray(new String[0]);
	}

	@Override
	public void invalidate() {
		httpSession.invalidate();
	}

	@Override
	public boolean isNew() {
		return httpSession.isNew();
	}

	@Override
	public void putValue(String name, Object value) {
		setAttributeInternal(name, value);
	}

	@Override
	public void removeAttribute(String name) {
		removeAttributeInternal(name);
	}

	@Override
	public void removeValue(String name) {
		removeAttributeInternal(name);
	}

	@Override
	public void setAttribute(String name, Object value) {
		setAttributeInternal(name, value);
	}

	@Override
	public void setMaxInactiveInterval(int interval) {
		httpSession.setMaxInactiveInterval(interval);
	}

	@Override
	public void setSharedAttribute(String name, Object value) {
		httpSession.setAttribute(name, value);
		purgePrivateAttributes(name);
	}

}
