package name.matthewgreet.strutscommons.interceptor;


import java.util.Map;
import java.util.Set;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.conversion.impl.ConversionData;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;

import name.matthewgreet.strutscommons.action.FormDriven;
import name.matthewgreet.strutscommons.form.NullForm;


/**
 * <P>If the action implements {@link FormDriven}, stores the form in the session so it may be retrieved by a later 
 * action that displays it.  It is usually part of the aflUpdateStack interceptor stack and retrieved by 
 * {@link FormRetrieveInterceptor} as part of the aflViewStack interceptor stack.</P>
 * 
 * 
 * <H2>Interceptor parameters</H2>
 * <DL>
 *  <DT>disabled</DT>
 *  <DD>If true, this interceptor is skipped.  Defaults to false.</DD>
 * </DL>
 * 
 * <H2>Extending the interceptor</H2>
 * <P>This can't be usefully extended.</P>
 * 
 * <H2>Example code</H2>
 * <PRE>
 * &#064;InterceptorRefs({
 *   &#064;InterceptorRef(value="formStore"),
 *   &#064;InterceptorRef(value="basicStack")
 *})
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class FormStoreInterceptor extends AbstractInterceptor {
    public static class StoredForm {
        private Object form;
        private Map<String, ConversionData> conversionErrors;
        private boolean invalid;
        private Class<?> processor;
        
        private String owningURL;
        private Class<?> owningActionClass;
        private Set<String> owningActionFieldNames;
        
        public Object getForm() {
            return form;
        }
        public void setForm(Object form) {
            this.form = form;
        }
        
		public Map<String, ConversionData> getConversionErrors() {
			return conversionErrors;
		}
		public void setConversionErrors(Map<String, ConversionData> conversionErrors) {
			this.conversionErrors = conversionErrors;
		}
        
        public boolean getInvalid() {
            return invalid;
        }
        public void setInvalid(boolean invalid) {
            this.invalid = invalid;
        }
        
        public Class<?> getProcessor() {
            return processor;
        }
        public void setProcessor(Class<?> processor) {
            this.processor = processor;
        }

        /**
         * Returns URL of page that retrieved messages so refresh still gets them. 
         */
        public String getOwningURL() {
            return owningURL;
        }
        public void setOwningURL(String value) {
            this.owningURL = value;
        }
        
        /**
         * Returns class of Action that retrieved form, or null if form not yet retrieved.
         */
        public Class<?> getOwningActionClass() {
            return owningActionClass;
        }
        public void setOwningActionClass(Class<?> owningActionClass) {
            this.owningActionClass = owningActionClass;
        }
        
        /**
         * Returns names of fields that received form, or null if form not yet retrieved.
         */
        public Set<String> getOwningActionFieldNames() {
            return owningActionFieldNames;
        }
        public void setOwningActionFieldNames(Set<String> owningActionFieldNames) {
            this.owningActionFieldNames = owningActionFieldNames;
        }
    }

    private static final long serialVersionUID = -2270259451208908292L;

    public static final String SESSION_STORED_FORM  = FormStoreInterceptor.class + "_STORED_FORM";
    
    private boolean disabled;

    
    public FormStoreInterceptor() {
        // Empty
    }

    
    
    @Override
    public String intercept(ActionInvocation invocation) throws Exception {
        FormDriven<?> formDriven;
        StoredForm storedForm;
        String result;
        
        result = invocation.invoke();
        if (!disabled && invocation.getAction() instanceof FormDriven) {
            formDriven = (FormDriven<?>)invocation.getAction();
            if (formDriven.getModel() != null && formDriven.getModel().getClass() != NullForm.class) {
                storedForm = new StoredForm();
                storedForm.setForm(formDriven.getModel());
                storedForm.setConversionErrors(invocation.getInvocationContext().getConversionErrors());
                storedForm.setInvalid(formDriven.formValidationFailed());
                storedForm.setProcessor(invocation.getAction().getClass());
                ActionContext.getContext().getSession().put(SESSION_STORED_FORM, storedForm);
            } else {
                ActionContext.getContext().getSession().put(SESSION_STORED_FORM, null);
            }
        }
        return result;
    }

    
	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	
}
