package name.matthewgreet.strutscommons.util;

import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import com.opensymphony.xwork2.interceptor.ValidationAware;

/**
 * Static library functions for formatting data for display but also parsing them for validation. 
 */
@SuppressWarnings("deprecation")
public class FormatLibrary {
    private static final String DATE_FORMAT_LONG_YEAR = "dd/MM/yyyy";
    private static final String DATETIME_FORMAT_SHORT_TIME = "dd/MM/yy HH:mm";
	private static final String NUMBER_FORMAT = "##,###,###";
	
	/**
     * Returns formatted date in DD/MM/YYYY without time component.
     */
    public static synchronized String formatDateLongYear(Date value) {
        DateFormat formatter;
        
        if (value != null) {
            formatter = new SimpleDateFormat(DATE_FORMAT_LONG_YEAR);
            return formatter.format(value);
        } else {
            return "";
        }
    }
    
	/**
     * Returns formatted date and time in DD/MM/YY HH:MM.
     */
    public static synchronized String formatDateShortTime(Date value) {
        DateFormat formatter;
        
        if (value != null) {
            formatter = new SimpleDateFormat(DATETIME_FORMAT_SHORT_TIME);
            return formatter.format(value);
        } else {
            return "";
        }
    }
    
	public static String formatNumber(Integer value) {
		if (value != null) {
			return (new DecimalFormat(NUMBER_FORMAT)).format(value);
		} else {
			return "";
		}
	}
	
	public static<T extends Enum<T>> T parseRequiredEnum(String field, Class<T> enumType,
            ValidationAware validationAware, String requiredMessage, String notEnumMessage) {
        T result = null;
        
        if (field == null || field.length() == 0) {
            validationAware.addActionError(requiredMessage);
        } else {
            try {
                result = Enum.valueOf(enumType, field);
            }
            catch (IllegalArgumentException e) {
                validationAware.addActionError(notEnumMessage);
            }
        }
        return result;
    }
    

	public static Integer parseInteger(String value, ValidationAware validationAware, String notNumberMessage) {
		if (value != null && value.trim().length() > 0) {
			try {
				return Integer.parseInt(value);
			}
			catch (NumberFormatException e) {
				validationAware.addActionError(notNumberMessage);
			}
		}
		return null;
	}

    public static int parseRequiredInt(String field, ValidationAware validationAware, 
            String requiredMessage, String notIntMessage) {
        int result = 0;
        
        if (field == null || field.trim().length() == 0) {
            validationAware.addActionError(requiredMessage);
        } else {
            try {
				result = Integer.parseInt(field);
            }
            catch (NumberFormatException e) {
                validationAware.addActionError(notIntMessage);
            }
        }
        return result;
    }

    public static String parseRequiredString(String field, ValidationAware validationAware, String requiredMessage) {
        String result = "";
        
        if (field == null || field.trim().length() == 0) {
            validationAware.addActionError(requiredMessage);
        } else {
		    result = field;
        }
        return result;
    }


}
