package name.matthewgreet.strutscommons.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Template that aids creation of OPTION tags in JSPs for single selection SELECT elements by formatting list for 
 * display.  Concrete subclasses override {@link #getValue} and {@link #getText} to extract item values and display text
 * from a raw list.  See also {@link SelectBoxItemDisplay}.
 * 
 * <P>Example JSP code:<BR/>
 * <CODE>
 * &lt;SELECT SIZE="1" NAME="category" /&gt;<BR/>
 * &nbsp;&nbsp;&lt;s:iterator value="categoryDisplay.list" var="categoryItemDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTION VALUE="&lt;s:property value="#categoryItemDisplay.value"/&gt;" &lt;s:property value="#categoryItemDisplay.selectedAttribute"/&gt;&gt;&lt;s:property value="#categoryItemDisplay.text"/&gt;&lt;/OPTION&gt;<BR/>
 * &nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &lt;/SELECT&gt;<BR/>
 * </CODE></P>
 * 
 * @deprecated Designed for forms using formatted/unformatted field pairs (pair conversion mode) and less useful for 
 * form fields set using auto conversion or default conversion mode.  Use {@link SingleSelectBoxDisplay2} as the base 
 * template and {@link SingleSelectBoxDisplay2#setSelectedFormattedValue} to set the selected item from the formatted 
 * half of a formatted/unformatted field pair.
 */
@Deprecated
public abstract class SingleSelectBoxDisplay<T> {
    public class TextComparator implements Comparator<SelectBoxItemDisplay<T>> {
        @Override
        public int compare(SelectBoxItemDisplay<T> o1, SelectBoxItemDisplay<T> o2) {
            return o1.getText().compareToIgnoreCase(o2.getText());
        }
    }

    private static Logger LOG = LogManager.getLogger(SingleSelectBoxDisplay.class);
    
    private List<SelectBoxItemDisplay<T>> formattedModel;
    private int selectedIndex;
    
    public SingleSelectBoxDisplay() {
        formattedModel = getInitialList();
        if (formattedModel == null) {
            formattedModel = new ArrayList<SelectBoxItemDisplay<T>>();
        }
        selectedIndex = -1;
    }

    /**
     * May be overridden by subclasses to add additional, formatted items to the display list.  This is typically used 
     * to add a blank item to represent no value or text that demands a real value is selected.  This is called after 
     * sorting.
     * 
     * @param formattedModel List of {@link SelectBoxItemDisplay}
     */
    protected void addItems(List<SelectBoxItemDisplay<T>> formattedModel) {
        // Empty
    }
    
    /**
     * If {@link #hasBlankValue} returns true or {@link #addBlankValue} is called, returns text to display for blank 
     * value.  Defaults to blank.
     */
    protected String getBlankText() {
        return "";
    }
    
    /**
     * May be overridden by subclasses to return initial selection, especially if the list is hardcoded and setModel 
     * will not be used.  Can return null for no initial list.
     */
    protected List<SelectBoxItemDisplay<T>> getInitialList() {
        return null;
    }
    
    /**
     * May be overridden by subclasses to return logger of subclass.
     */
    protected Logger getLogger() {
        return LOG;
    }
    
    /**
     * May be overridden by subclasses to return a comparator for defining display order of formatted items.  Comparator
     * compares instances of {@link SelectBoxItemDisplay} that are allowed by subclass.  Defaults to comparator that 
     * sorts by ascending display text. 
     */
    protected Comparator<SelectBoxItemDisplay<T>> getSortComparator() {
        return new TextComparator();
    }
    
    /**
     * Overridden by subclasses to return formatted text to be displayed to user as part of OPTION element.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getText(T item);
    
    /**
     * Overridden by subclasses to return string to be used in VALUE attribute of OPTION element.  This is the value 
     * return in form data. 
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getValue(T item);
    
    /**
     * May be overridden by subclasses to return true, inserting a blank value at the beginning of the SELECT list.
     * Defaults to false. 
     */
    protected boolean hasBlankValue() {
        return false;
    }
    
    /**
     * May be overridden by subclasses to filter items in lookup list from display.  Defaults to allow all items.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected boolean isAllowed(T item) {
        return true;
    }
    
    /**
     * If called after {@link #setModel}, inserts blank value to beginning of list.  This is useful for search fields 
     * where a blank value means no search.
     */
    public void addBlankValue() {
        SelectBoxItemDisplay<T> formattedItem;
        
        formattedItem = new SelectBoxItemDisplay<T>("", getBlankText(), null);
        formattedModel.add(0, formattedItem);
    }
    
    /**
     * Returns formatted version of account list for human display.
     */
    public List<SelectBoxItemDisplay<T>> getList() {
        return formattedModel;
    }
    
    /**
     * Returns selected, formatted item or null if none selected. 
     */
    public SelectBoxItemDisplay<T> getSelectedItem() {
        if (selectedIndex > -1) {
            return formattedModel.get(selectedIndex);
        } else {
            return null;
        }
    }
    
    /**
     * Directly sets formatted version of account list for human display.  The value must already be sorted.
     */
    public void setList(List<SelectBoxItemDisplay<T>> value) {
        formattedModel = value;
    }
    
    /**
     * Formats lookup list for use in single selection HTML SELECT element.  This replaces any initial list.
     */
    public void setModel(Collection<T> model) {
        SelectBoxItemDisplay<T> formattedItem;
        String value, text;
        
        try {
            formattedModel.clear();
            if (model != null) {
                for (T item: model) {
                    if (isAllowed(item)) {
                        value = getValue(item);
                        text = getText(item);
                        formattedItem = new SelectBoxItemDisplay<T>(value, text, item);
                        formattedModel.add(formattedItem);
                    }
                }
            }
            
            Collections.sort(formattedModel, getSortComparator());
            
            addItems(formattedModel);
            if (hasBlankValue()) {
                formattedModel.add(0, new SelectBoxItemDisplay<T>("", getBlankText(), null));
            }
        }
        catch (Exception e) {
            LOG.error("Setup of JSP view helper failed", e);
        }
    }

    /**
     * Sets selected item of list identified by its value.
     * 
     * @param value id of selected item or null or empty string to select none.
     */
    public void setSelectedValue(String value) {
        SelectBoxItemDisplay<T> item;
        ListIterator<SelectBoxItemDisplay<T>> iter;
        boolean found;
        
        // Deselect existing, selected item.
        if (selectedIndex > -1) {
            item = formattedModel.get(selectedIndex);
            item.setSelected(false);
        }
        selectedIndex = -1;
        
        if (value != null && value.length() > 0) {
            // Find item and set it as selected
            iter = formattedModel.listIterator();
            found = false;
            while (iter.hasNext() && !found) {
                item = iter.next();
                if (item.getValue().equals(value)) {
                    selectedIndex = iter.previousIndex();
                    item.setSelected(true);
                    found = true;
                }
            }
        }
    }
}
