package name.matthewgreet.strutscommons.junit.policy;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Locale;
import java.util.TimeZone;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

import com.opensymphony.xwork2.ActionContext;

import name.matthewgreet.strutscommons.annotation.DateConversion;
import name.matthewgreet.strutscommons.annotation.DateConversion.Style;
import name.matthewgreet.strutscommons.policy.ConversionResult;
import name.matthewgreet.strutscommons.policy.DateConverter;


/**
 * Units tests for DateConverter.  
 */
@SuppressWarnings("deprecation")
public class DateConverterUnitTests {
	@DateConversion
	private boolean annotationPlaceholderDefault;
	
	@DateConversion(format = "yyyy-MM-dd'T'HH:mm:ss.SSS")
	private boolean annotationPlaceholderISO8601;
	
	@DateConversion(format = "", localeCountry = "LU", localeLanguage = "DE")
	private boolean annotationPlaceholderLocale;
	
	@DateConversion(format = "", style = Style.MEDIUM)
	private boolean annotationPlaceholderStyle;
	
	@BeforeEach
	public void init() {
		// Empty
	}
	
	
	@DisplayName("Normal conversion")
	@Test
	public void testConvert1() throws Exception {
		Calendar calendar;
		DateConversion annotation;
		DateConverter dateConverter;
		ConversionResult<Date> result;
		
		calendar = new GregorianCalendar();
		calendar.set(1945, 2, 8, 0, 0, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.convert("08/03/1945", Date.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(calendar.getTime(), result.getParsedValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Failed conversion")
	@Test
	public void testConvert2() throws Exception {
		DateConversion annotation;
		DateConverter dateConverter;
		ConversionResult<Date> result;
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.convert("38/3/1945", Date.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(null, result.getMessage(), "Standard policies leave message to annotation");
		Assertions.assertEquals(null, result.getMessageKey(), "Standard policies leave message key to annotation");
		Assertions.assertEquals(null, result.getMessageType(), "Standard policies leave message type to annotation");
		Assertions.assertEquals(false, result.getSuccess(), "Conversion should have failed");
	}
    
	@DisplayName("ISO 8601 conversion, includes time")
	@Test
	public void testConvert3() throws Exception {
		Calendar calendar;
		DateConversion annotation;
		DateConverter dateConverter;
		ConversionResult<Date> result;
		
		calendar = new GregorianCalendar();
		calendar.set(2067, 11, 19, 15, 39, 42);
		calendar.set(Calendar.MILLISECOND, 848);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderISO8601").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.convert("2067-12-19T15:39:42.848", Date.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(calendar.getTime(), result.getParsedValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("German Luxembourg date format")
	@Test
	public void testConvert4() throws Exception {
		Calendar calendar;
		DateConversion annotation;
		DateConverter dateConverter;
		ConversionResult<Date> result;
		
		calendar = new GregorianCalendar();
		calendar.set(1991, 9, 18, 0, 0, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderLocale").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.convert("18.10.1991", Date.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(calendar.getTime(), result.getParsedValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
	@DisplayName("Medium style, ActionContext hardcoded to German Luxembourg")
	@Test
	public void testConvert5() throws Exception {
		ActionContext actionContext;
		Calendar calendar;
		DateConversion annotation;
		DateConverter dateConverter;
		ConversionResult<Date> result;
		
		calendar = new GregorianCalendar();
		calendar.set(1923, 6, 24, 0, 0, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		actionContext = ActionContext.bind(ActionContext.of());
		actionContext.withLocale(new Locale("DE", "LU"));
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderStyle").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.convert("24.07.1923", Date.class);
		Assertions.assertNotNull(result, "Converters never return null");
		Assertions.assertEquals(calendar.getTime(), result.getParsedValue(), "Final value is incorrect");
		Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
	}
    
    @DisplayName("ISO 8601 conversion, includes time but default time zone is PST")
    @Test
    public void testConvert6() throws Exception {
        Calendar calendar;
        DateConversion annotation;
        DateConverter dateConverter;
        ConversionResult<Date> result;
        TimeZone defaultTimeZone;
        
        defaultTimeZone = TimeZone.getDefault();
        TimeZone.setDefault(TimeZone.getTimeZone("PST"));
        
        try {
            calendar = new GregorianCalendar();
            calendar.set(2067, 11, 19, 15, 39, 42);
            calendar.set(Calendar.MILLISECOND, 848);
            calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
            
            annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderISO8601").getAnnotation(DateConversion.class);
            dateConverter = new DateConverter();
            dateConverter.setAnnotation(annotation);
            result = dateConverter.convert("2067-12-19T15:39:42.848", Date.class);
            Assertions.assertNotNull(result, "Converters never return null");
            Assertions.assertEquals(calendar.getTime(), result.getParsedValue(), "Final value is incorrect");
            Assertions.assertEquals(true, result.getSuccess(), "Conversion should be successful");
        }
        finally {
            TimeZone.setDefault(defaultTimeZone);
        }
    }
    
	@DisplayName("Normal format")
	@Test
	public void testFormat1() throws Exception {
		Calendar calendar;
		DateConversion annotation;
		DateConverter dateConverter;
		String result;
		
		calendar = new GregorianCalendar();
		calendar.set(1945, 2, 8, 0, 0, 0);
		calendar.set(Calendar.MILLISECOND, 0);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		annotation = (DateConversion)DateConverterUnitTests.class.getDeclaredField("annotationPlaceholderDefault").getAnnotation(DateConversion.class);
		dateConverter = new DateConverter();
		dateConverter.setAnnotation(annotation);
		result = dateConverter.format(calendar.getTime());
		Assertions.assertEquals("08/03/1945", result, "Final value is incorrect");
	}
    
}
